/*****************************************************************************
 *
 * This document contains proprietary information and except with
 * written permission of Displaytech, Ltd., such information shall
 * not be published or disclosed to others or used for any purpose
 * other than for the operation and maintenance of the equipment
 * and software with which it was procured, and the document shall
 * not be copied in whole or in part.
 *
 * Copyright 2008-2014 Displaytech, Ltd. All Rights Reserved.
 *
 *****************************************************************************
 *  Solomon Systech. SSD1963 LCD controller driver
 *****************************************************************************
 * FileName:        SSD1963.h
 * Processor:       PIC32
 * Compiler:        XC32
 * Company:         Displaytech Ltd.
 *
 * Date             Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 11/29/2012       Initial Revision
 *****************************************************************************/

#ifndef _SSD1963_H
#define _SSD1963_H

#include "GenericTypeDefs.h"

/*********************************************************************
* Overview: Some basic colors definitions.
*********************************************************************/
#define BLACK               RGB565CONVERT(0,    0,      0)
#define BRIGHTBLUE          RGB565CONVERT(0,    0,      255)
#define BRIGHTGREEN         RGB565CONVERT(0,    255,    0)
#define BRIGHTCYAN          RGB565CONVERT(0,    255,    255)
#define BRIGHTRED           RGB565CONVERT(255,  0,      0)
#define BRIGHTMAGENTA       RGB565CONVERT(255,  0,      255)
#define BRIGHTYELLOW        RGB565CONVERT(255,  255,    0)
#define BLUE                RGB565CONVERT(0,    0,      128)
#define GREEN               RGB565CONVERT(0,    128,    0)
#define CYAN                RGB565CONVERT(0,    128,    128)
#define RED                 RGB565CONVERT(128,  0,      0)
#define MAGENTA             RGB565CONVERT(128,  0,      128)
#define BROWN               RGB565CONVERT(255,  128,    0)
#define LIGHTGRAY           RGB565CONVERT(128,  128,    128)
#define DARKGRAY            RGB565CONVERT(64,   64,     64)
#define LIGHTBLUE           RGB565CONVERT(128,  128,    255)
#define LIGHTGREEN          RGB565CONVERT(128,  255,    128)
#define LIGHTCYAN           RGB565CONVERT(128,  255,    255)
#define LIGHTRED            RGB565CONVERT(255,  128,    128)
#define LIGHTMAGENTA        RGB565CONVERT(255,  128,    255)
#define YELLOW              RGB565CONVERT(255,  255,    128)
#define WHITE               RGB565CONVERT(255,  255,    255)

#define GRAY0               RGB565CONVERT(224,  224,    224)
#define GRAY1               RGB565CONVERT(192,  192,    192)
#define GRAY2               RGB565CONVERT(160,  160,    160)
#define GRAY3               RGB565CONVERT(128,  128,    128)
#define GRAY4               RGB565CONVERT(96,   96,     96)
#define GRAY5               RGB565CONVERT(64,   64,     64)
#define GRAY6               RGB565CONVERT(32,   32,     32)

/*********************************************************************
* SSD1961 command table
*********************************************************************/
#define CMD_NOP                     0x00
#define CMD_SOFT_RESET              0x01
#define CMD_GET_PWR_MODE            0x0A
#define CMD_GET_ADDR_MODE           0x0B
#define CMD_GET_DISPLAY_MODE        0x0D
#define CMD_GET_TEAR_EFFECT_STATUS  0x0E
#define CMD_ENT_SLEEP               0x10
#define CMD_EXIT_SLEEP              0x11
#define CMD_ENT_PARTIAL_MODE        0x12
#define CMD_ENT_NORMAL_MODE         0x13
#define CMD_EXIT_INVERT_MODE        0x20
#define CMD_ENT_INVERT_MODE         0x21
#define CMD_SET_GAMMA               0x26
#define CMD_BLANK_DISPLAY           0x28
#define CMD_ON_DISPLAY              0x29
#define CMD_SET_COLUMN              0x2A
#define CMD_SET_PAGE                0x2B
#define CMD_WR_MEMSTART             0x2C
#define CMD_RD_MEMSTART             0x2E
#define CMD_SET_PARTIAL_AREA        0x30
#define CMD_SET_SCROLL_AREA         0x33
#define CMD_SET_TEAR_OFF            0x34
#define CMD_SET_TEAR_ON             0x35
#define CMD_SET_ADDR_MODE           0x36
#define CMD_SET_SCROLL_START        0x37
#define CMD_EXIT_IDLE_MODE          0x38
#define CMD_ENT_IDLE_MODE           0x39
#define CMD_WR_MEM_AUTO             0x3C
#define CMD_RD_MEM_AUTO             0x3E
#define CMD_SET_TEAR_SCANLINE       0x44
#define CMD_GET_SCANLINE            0x45
#define CMD_RD_DDB_START            0xA1
#define CMD_SET_PANEL_MODE          0xB0
#define CMD_GET_PANEL_MODE          0xB1
#define CMD_SET_HOR_PERIOD          0xB4
#define CMD_GET_HOR_PERIOD          0xB5
#define CMD_SET_VER_PERIOD          0xB6
#define CMD_GET_VER_PERIOD          0xB7
#define CMD_SET_GPIO_CONF           0xB8
#define CMD_GET_GPIO_CONF           0xB9
#define CMD_SET_GPIO_VAL            0xBA
#define CMD_GET_GPIO_STATUS         0xBB
#define CMD_SET_POST_PROC           0xBC
#define CMD_GET_POST_PROC           0xBD
#define CMD_SET_PWM_CONF            0xBE
#define CMD_GET_PWM_CONF            0xBF
#define CMD_SET_LCD_GEN0            0xC0
#define CMD_GET_LCD_GEN0            0xC1
#define CMD_SET_LCD_GEN1            0xC2
#define CMD_GET_LCD_GEN1            0xC3
#define CMD_SET_LCD_GEN2            0xC4
#define CMD_GET_LCD_GEN2            0xC5
#define CMD_SET_LCD_GEN3            0xC6
#define CMD_GET_LCD_GEN3            0xC7
#define CMD_SET_GPIO0_ROP           0xC8
#define CMD_GET_GPIO0_ROP           0xC9
#define CMD_SET_GPIO1_ROP           0xCA
#define CMD_GET_GPIO1_ROP           0xCB
#define CMD_SET_GPIO2_ROP           0xCC
#define CMD_GET_GPIO2_ROP           0xCD
#define CMD_SET_GPIO3_ROP           0xCE
#define CMD_GET_GPIO3_ROP           0xCF
#define CMD_SET_ABC_DBC_CONF        0xD0
#define CMD_GET_ABC_DBC_CONF        0xD1
#define CMD_SET_DBC_THRES           0xD4
#define CMD_GET_DBC_THRES           0xD5
#define CMD_PLL_START               0xE0
#define CMD_SET_PLL_MN              0xE2
#define CMD_GET_PLL_MN              0xE3
#define CMD_GET_PLL_STATUS          0xE4
#define CMD_ENT_DEEP_SLEEP          0xE5
#define CMD_SET_PCLK                0xE6
#define CMD_GET_PCLK                0xE7
#define CMD_SET_DATA_INTERFACE      0xF0
#define CMD_GET_DATA_INTERFACE      0xF1


/*********************************************************************
* Function:  void ResetDevice()
*
* Overview: Initializes LCD module.
*
* PreCondition: none
*
* Input: none
*
* Output: none
*
* Side Effects: none
*
********************************************************************/
void ResetDevice(void);


/*********************************************************************
* Function: void PutPixel(SHORT x, SHORT y)
*
* Overview: Puts pixel with the given x,y coordinate position.
*
* PreCondition: none
*
* Input: x - x position of the pixel.
*   y - y position of the pixel.
*
* Output: none
*
* Side Effects: none
*
********************************************************************/
void PutPixel(SHORT x, SHORT y);

/*********************************************************************
* Function: void ClearDevice(void)
*
* PreCondition: none
*
* Input: none
*
* Output: none
*
* Side Effects: none
*
* Overview: clears screen with current color
*
* Note: none
*
********************************************************************/
void ClearDevice(void);

/*********************************************************************
* Function:  SetScrollArea(SHORT top, SHORT scroll, SHORT bottom)
*
* PreCondition: none
*
* Input: top - Top Fixed Area in number of lines from the top
*    of the frame buffer
*   scroll - Vertical scrolling area in number of lines
*   bottom - Bottom Fixed Area in number of lines
*
* Output: none
*
* Side Effects: none
*
* Overview:
*
* Note: Reference: section 9.22 Set Scroll Area, SSD1963 datasheet Rev0.20
*
********************************************************************/
void SetScrollArea(SHORT top, SHORT scroll, SHORT bottom);

/*********************************************************************
* Function:  void  SetScrollStart(SHORT line)
*
* Overview: First, we need to define the scrolling area by SetScrollArea()
*   before using this function.
*
* PreCondition: SetScrollArea(SHORT top, SHORT scroll, SHORT bottom)
*
* Input: line - Vertical scrolling pointer (in number of lines) as
*   the first display line from the Top Fixed Area defined in SetScrollArea()
*
* Output: none
*
* Note: none
********************************************************************/
void SetScrollStart(SHORT line);

/*********************************************************************
* Function:  void EnterSleepMode (void)
* PreCondition: none
* Input:  none
* Output: none
* Side Effects: none
* Overview: SSD1963 enters sleep mode
* Note: Host must wait 5mS after sending before sending next command
********************************************************************/
void EnterSleepMode (void);

/*********************************************************************
* Function:  void ExitSleepMode (void)
* PreCondition: none
* Input:  none
* Output: none
* Side Effects: none
* Overview: SSD1963 exits sleep mode
* Note:   cannot be called sooner than 15ms
********************************************************************/
void ExitSleepMode (void);

/*********************************************************************
* Function  : void DisplayOff(void)
* PreCondition : none
* Input   : none
* Output  : none
* Side Effects : none
* Overview  : SSD1963 changes the display state to OFF state
* Note   : none
********************************************************************/
void DisplayOff(void);

/*********************************************************************
* Function  : void DisplayOn(void)
* PreCondition : none
* Input   : none
* Output  : none
* Side Effects : none
* Overview  : SSD1963 changes the display state to ON state
* Note   : none
********************************************************************/
void DisplayOn(void);

/*********************************************************************
* Function  : void EnterDeepSleep(void)
* PreCondition : none
* Input   : none
* Output  : none
* Side Effects : none
* Overview  : SSD1963 enters deep sleep state with PLL stopped
* Note   : none
********************************************************************/
void EnterDeepSleep(void);

/*********************************************************************
* Function:  void  SetBacklight(BYTE intensity)
*
* Overview: This function makes use of PWM feature of ssd1963 to adjust
*   the backlight intensity.
*
* PreCondition: Backlight circuit with shutdown pin connected to PWM output of ssd1963.
*
* Input:  (BYTE) intensity from
*   0x00 (total backlight shutdown, PWM pin pull-down to VSS)
*   0xff (99% pull-up, 255/256 pull-up to VDD)
*
* Output: none
*
* Note: The base frequency of PWM set to around 300Hz with PLL set to 120MHz.
*  This parameter is hardware dependent
********************************************************************/
void SetBacklight(BYTE intensity);

/*********************************************************************
* Function:  void  SetTearingCfg(BOOL state, BOOL mode)
*
* Overview: This function enable/disable tearing effect
*
* PreCondition: none
*
* Input:  BOOL state - 1 to enable
*       0 to disable
*   BOOL mode -  0:  the tearing effect output line consists
*        of V-blanking information only
*       1: the tearing effect output line consists
*        of both V-blanking and H-blanking info.
* Output: none
*
* Note:
********************************************************************/
void SetTearingCfg(BOOL state, BOOL mode);


#endif // _SSD1963_H
