/*****************************************************************************
 *
 * This document contains proprietary information and except with
 * written permission of Displaytech, Ltd., such information shall
 * not be published or disclosed to others or used for any purpose
 * other than for the operation and maintenance of the equipment
 * and software with which it was procured, and the document shall
 * not be copied in whole or in part.
 *
 * Copyright 2008-2014 Displaytech, Ltd. All Rights Reserved.
 *
 *****************************************************************************
 *  Solomon Systech. SSD1961 LCD controller driver
 *****************************************************************************
 * FileName:        SSD1961.c
 * Processor:       PIC32
 * Compiler:        XC32
 * Company:         Displaytech Ltd.
 *
 * Date             Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 11/29/2012       Initial Revision
 *****************************************************************************/

#include "SSD1961.h"
#include "HardwareProfile.h"
#include "Compiler.h"
#include "TimeDelay.h"
#include "Graphics/DisplayDriver.h"
#include "Graphics/gfxtcon.h"
#include "Graphics/Primitive.h"
#if defined (USE_GFX_PMP)
#include "Graphics/gfxpmp.h"
#elif defined (USE_GFX_EPMP)
#include "Graphics/gfxepmp.h"
#endif

/******************************** GLOBAL VARIABLE ********************************/
// Color
GFX_COLOR _color;
#ifdef USE_TRANSPARENT_COLOR
GFX_COLOR _colorTransparent;
SHORT _colorTransparentEnable;
#endif

// Clipping region control
SHORT _clipRgn;

// Clipping region borders
SHORT _clipLeft;
SHORT _clipTop;
SHORT _clipRight;
SHORT _clipBottom;

/**************** LOCAL FUNCTION PROTOTYPE (SSD1961 SPECIFIC) ****************/
static void SetArea(SHORT start_x, SHORT start_y, SHORT end_x, SHORT end_y);
static void GPIO_WR(BYTE pin, BOOL state);

/*********************************************************************
 * Macros:  WriteCommand(cmd)
 *
 * PreCondition:
 *
 * Input: cmd - controller command
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: writes command
 *
 * Note: none
 ********************************************************************/
#define WriteCommand(cmd) {DisplayEnable();     \
                           DisplaySetCommand(); \
                           DeviceWrite(cmd) ;   \
                           DisplayDisable();}

/*********************************************************************
 * Function:  void  WriteData(WORD data)
 *
 * PreCondition:
 *
 * Input:  value - value to be written in WORD format
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview:
 *
 * Note: none
 ********************************************************************/
#define WriteData(data) {DisplaySetData(); DeviceWrite(data);}

/*********************************************************************
 * Function:  void  ReadData(WORD data)
 *
 * PreCondition:
 *
 * Input: none
 *
 * Output: The 16-bit data read
 *
 * Side Effects: none
 *
 * Overview:
 *
 * Note: none
 ********************************************************************/
WORD ReadData(void) {
    WORD data;

    DisplaySetData();
    data = DeviceRead();
}

/*********************************************************************
 * Macros:  ReadRegister(cmd)
 *
 * PreCondition:
 *
 * Input: cmd - register
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Reads a register value
 *
 * Note: Since Microchip's PMP module is used, the syntax PMDIN = cmd
 *  is used. For other MCUs, this could be PORTx=cmd that PORTx
 *  is the data bus from your MCU.
 ********************************************************************/
WORD ReadRegister(WORD reg) {
    WORD data;
    DisplayEnable();
    DisplaySetCommand();
    DeviceWrite(reg);
    DisplaySetData();
    data = DeviceRead();
    DisplayDisable();
    return data;
}

/*********************************************************************
 * Function:  SetArea(start_x,start_y,end_x,end_y)
 *
 * PreCondition: SetActivePage(page)
 *
 * Input: start_x, end_x - start column and end column
 *   start_y,end_y  - start row and end row position (i.e. page address)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: defines start/end columns and start/end rows for memory access
 *   from host to SSD1961
 * Note: none
 ********************************************************************/
void SetArea(SHORT start_x, SHORT start_y, SHORT end_x, SHORT end_y) {
    WriteCommand(CMD_SET_COLUMN);
    DisplayEnable();
    WriteData(start_x >> 8);
    WriteData(start_x);
    WriteData(end_x >> 8);
    WriteData(end_x);
    DisplayDisable();
    WriteCommand(CMD_SET_PAGE);
    DisplayEnable();
    WriteData(start_y >> 8);
    WriteData(start_y);
    WriteData(end_y >> 8);
    WriteData(end_y);
    DisplayDisable();
}

/*********************************************************************
 * Function: Set a GPIO pin to state high(1) or low(0)
 *
 * PreCondition: Set the GPIO pin an output prior using this function
 *
 * Arguments: BYTE pin -  LCD_RESET
 *       LCD_SPENA
 *       LCD_SPCLK
 *       LCD_SPDAT defined under GraphicsConfig.h
 *
 *    BOOL state -  0 for low
 *       1 for high
 * Return: none
 *
 * Note:
 *********************************************************************/
static void GPIO_WR(BYTE pin, BOOL state) {
    BYTE _gpioStatus = 0;

    if (state == 1)
        _gpioStatus = _gpioStatus | pin;
    else
        _gpioStatus = _gpioStatus & (~pin);

    WriteCommand(CMD_SET_GPIO_VAL); // Set GPIO value
    DisplayEnable();
    WriteData(_gpioStatus);
    DisplayDisable();
}

/*********************************************************************
 * Function:  void EnterSleepMode (void)
 * PreCondition: none
 * Input:  none
 * Output: none
 * Side Effects: none
 * Overview: SSD1961 enters sleep mode
 * Note: Host must wait 5mS after sending before sending next command
 ********************************************************************/
void EnterSleepMode(void) {
    WriteCommand(CMD_ENT_SLEEP);
}

/*********************************************************************
 * Function:  void ExitSleepMode (void)
 * PreCondition: none
 * Input:  none
 * Output: none
 * Side Effects: none
 * Overview: SSD1961 enters sleep mode
 * Note:   none
 ********************************************************************/
void ExitSleepMode(void) {
    WriteCommand(CMD_EXIT_SLEEP);
}

/*********************************************************************
 * Function  : void DisplayOff(void)
 * PreCondition : none
 * Input   : none
 * Output  : none
 * Side Effects : none
 * Overview  : SSD1961 changes the display state to OFF state
 * Note   : none
 ********************************************************************/
void DisplayOff(void) {
    WriteCommand(CMD_OFF_DISPLAY);
}

/*********************************************************************
 * Function  : void DisplayOn(void)
 * PreCondition : none
 * Input   : none
 * Output  : none
 * Side Effects : none
 * Overview  : SSD1961 changes the display state to ON state
 * Note   : none
 ********************************************************************/
void DisplayOn(void) {
    WriteCommand(CMD_ON_DISPLAY);
}

/*********************************************************************
 * Function  : void EnterDeepSleep(void)
 * PreCondition : none
 * Input   : none
 * Output  : none
 * Side Effects : none
 * Overview  : SSD1961 enters deep sleep state with PLL stopped
 * Note   : none
 ********************************************************************/
void EnterDeepSleep(void) {
    WriteCommand(CMD_ENT_DEEP_SLEEP);
}

/*********************************************************************
 * Function:  void  SetBacklight(BYTE intensity)
 *
 * Overview: This function makes use of PWM feature of ssd1961 to adjust
 *   the backlight intensity.
 *
 * PreCondition: Backlight circuit with shutdown pin connected to PWM output of ssd1961.
 *
 * Input:  (BYTE) intensity from
 *   0x00 (total backlight shutdown, PWM pin pull-down to VSS)
 *   0xff (99% pull-up, 255/256 pull-up to VDD)
 *
 * Output: none
 *
 * Note: The base frequency of PWM set to around 300Hz with PLL set to 120MHz.
 *  This parameter is hardware dependent
 ********************************************************************/
void SetBacklight(BYTE intensity) {
    WriteCommand(CMD_SET_PWM_CONF); // Set PWM configuration for backlight control
    DisplayEnable();
    WriteData(0x0E); // PWMF[7:0] = 2, PWM base freq = PLL/(256*(1+5))/256 =
    // 300Hz for a PLL freq = 120MHz
    WriteData(intensity); // Set duty cycle, from 0x00 (total pull-down) to 0xFF
    // (99% pull-up , 255/256)
    WriteData(0x01); // PWM enabled and controlled by host (mcu)
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x00);

    DisplayDisable();
}

/*********************************************************************
 * Function:  void  SetTearingCfg(BOOL state, BOOL mode)
 *
 * Overview: This function enable/disable tearing effect
 *
 * PreCondition: none
 *
 * Input:  BOOL state - 1 to enable
 *       0 to disable
 *   BOOL mode -  0:  the tearing effect output line consists
 *        of V-blanking information only
 *       1: the tearing effect output line consists
 *        of both V-blanking and H-blanking info.
 * Output: none
 *
 * Note:
 ********************************************************************/
void SetTearingCfg(BOOL state, BOOL mode) {
    if (state == 1) {
        WriteCommand(CMD_SET_TEAR_ON);
        DisplayEnable();
        WriteData(mode & 0x01);
        DisplayDisable();
    } else {
        WriteCommand(0x34);
    }

}

/*********************************************************************
 * Function:  void ResetDevice()
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Initializes PMP
 *   Initialize low level IO port for mcu,
 *   initialize SSD1961 for color-depth
 *
 * Note: none
 ********************************************************************/
void ResetDevice(void) {
    //Set up the pins
    DisplayCmdDataConfig(); // enable RS line
    DisplayDisable(); // not selected by default
    DisplayConfig(); // enable chip select line
    DisplayWrHigh();
    DisplayWrConfig();

    // PMP setup
    PMMODE = 0;
    PMAEN = 0;
    PMCON = 0;
    PMMODEbits.MODE = 2; // Intel 80 master interface
    PMMODEbits.WAITB = 3;
    PMMODEbits.WAITM = 15;
    PMMODEbits.WAITE = 3;
    PMMODEbits.MODE16 = 1; // 16 bit mode
    PMCONbits.PTRDEN = 1; // enable RD line
    PMCONbits.PTWREN = 1; // enable WR line
    PMCONbits.PMPEN = 1; // enable PMP

#if (COLOR_DEPTH == 16)
    //Set 16-bit 5-6-5
    WriteCommand(CMD_SET_DATA_INTERFACE);
    DisplayEnable();
    WriteData(0x03);
    DisplayDisable();
#elif (COLOR_DEPTH == 24)
    WriteCommand(CMD_SET_DATA_INTERFACE);
    DisplayEnable();
    WriteData(0x02);
    DisplayDisable();
#endif

    WriteCommand(CMD_ON_DISPLAY); // Turn on display; show the image on display
}

/*********************************************************************
 * Function: void PutPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: puts pixel
 *
 * Note:
 ********************************************************************/
void PutPixel(SHORT x, SHORT y) {
    static SHORT lastX = 0;
    static SHORT lastY = 0;

    //Don't always set the new position if we don't need to.
    if (!(lastY == y && x == ++lastX)) {
        SetArea(x, y, DISP_HOR_RESOLUTION - 1, DISP_VER_RESOLUTION - 1);
    WriteCommand(CMD_WR_MEMSTART);
        lastX = x;
        lastY = y;
    }

    DisplayEnable();
    DisplaySetData();
#if (COLOR_DEPTH == 16)
    DeviceWrite(_color);
#elif (COLOR_DEPTH == 24)
    DeviceWrite(_color >> 8);
    DeviceWrite((_color << 8) | (_color >> 16));
    DeviceWrite(_color);
#endif
    DisplayDisable();
}

/*********************************************************************
 * Function: GFX_COLOR GetPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: Pixel value
 *
 * Side Effects: none
 *
 * Overview: Gets pixel
 *
 * Note:
 ********************************************************************/
GFX_COLOR GetPixel(SHORT x, SHORT y) {
    GFX_COLOR pixel;

    SetArea(x, y, DISP_HOR_RESOLUTION - 1, DISP_VER_RESOLUTION - 1);
    WriteCommand(CMD_RD_MEMSTART);
    DisplayEnable();
    DisplaySetData();
#if COLOR_DEPTH == 16
    pixel = DeviceRead();
#elif COLOR_DEPTH == 24
    pixel = DeviceRead();
    pixel <<= 8;
    pixel |= DeviceRead();
    pixel |= (DeviceRead() & 0x00); //Ignore this read
#endif
    DisplayDisable();

    return pixel;
}

/*********************************************************************
 * Function: void GetPixelArray()
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data read
 *              yStart: The y starting position of the data read
 *              *desArray: The array of data to be read
 *              len: The length of the data to read
 *              dir: The direction of the read (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Reads an array of data of length len from the frame buffer
 *           on the SSD1961.  This function is used for alphablending.
 *
 * Note: none
 *
 ********************************************************************/
void GetPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *desArray, WORD len, WORD dir) {
    SetArea(xStart, yStart, DISP_HOR_RESOLUTION - 1, DISP_VER_RESOLUTION - 1);
    WriteCommand(CMD_RD_MEMSTART);
    DisplayEnable();
    DisplaySetData();
    while (len--) {
#if COLOR_DEPTH == 16
        *desArray = DeviceRead();
#elif COLOR_DEPTH == 24
        *desArray = DeviceRead();
        *desArray <<= 8;
        *desArray |= DeviceRead();
        *desArray |= (DeviceRead() & 0x00); //Ignore this read
#endif
        ++desArray;
    }
    DisplayDisable();
}

/*********************************************************************
 * Function: void PutPixelArray()
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data write
 *              yStart: The y starting position of the data write
 *              *desArray: The array of data to be written
 *              len: The length of the data to write
 *              dir: The direction of the write (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Writes an array of data of length len to the frame buffer
 *           on the SSD1961.  This function is used for alphablending
 *
 * Note: none
 *
 ********************************************************************/
void PutPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *srcArray, WORD len, WORD dir) {
    SetArea(xStart, yStart, DISP_HOR_RESOLUTION - 1, DISP_VER_RESOLUTION - 1);
    WriteCommand(CMD_WR_MEMSTART);
    DisplayEnable();
    DisplaySetData();
    while (len--) {
#ifdef USE_TRANSPARENT_COLOR
        if (_colorTransparentEnable == TRANSPARENT_COLOR_ENABLE) {
            if (_colorTransparent == *srcArray) {
                DisplayDisable();
                if (++xStart > GetMaxX()) {
                    xStart = 0;
                    if (++yStart > GetMaxY())
                        yStart = 0;
                }
                SetArea(xStart, yStart, GetMaxX(), GetMaxY());

                WriteCommand(CMD_WR_MEMSTART);
                DisplayEnable();
                DisplaySetData();

                ++srcArray;
                continue;
            }
        }
#endif

#if (COLOR_DEPTH == 16)
        DeviceWrite(*srcArray);
#elif (COLOR_DEPTH == 24)
        DeviceWrite(*srcArray >> 8);
        DeviceWrite((*srcArray << 8) | (*srcArray >> 16));
        DeviceWrite(*srcArray);
#endif
        ++srcArray;

#ifdef USE_TRANSPARENT_COLOR
        if (++xStart > GetMaxX()) {
            xStart = 0;
            if (++yStart > GetMaxY())
                yStart = 0;
        }
#endif
    }
    DisplayDisable();
}

/*********************************************************************
 * Function: WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom)
 *
 * PreCondition: none
 *
 * Input: left,top - top left corner coordinates,
 *        right,bottom - bottom right corner coordinates
 *
 * Output: For NON-Blocking configuration:
 *         - Returns 0 when device is busy and the shape is not yet completely drawn.
 *         - Returns 1 when the shape is completely drawn.
 *         For Blocking configuration:
 *         - Always return 1.
 *
 * Side Effects: none
 *
 * Overview: draws rectangle filled with current color
 *
 * Note: none
 *
 ********************************************************************/
WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom) {

    if(left > right)
    {
        return (1); /* Don't draw but return 1 */
    }

    if(top > bottom)
    {
        return (1); /* Don't draw but return 1 */
    }

    DWORD counter = ((bottom + 1) - top)*((right + 1) - left);
#ifdef USE_ALPHABLEND_LITE
    if (GetAlpha() != 100) {
        BarAlpha(left, top, right, bottom);
        return 1;
    }
#endif
    SetArea(left, top, right, bottom);

    WriteCommand(CMD_WR_MEMSTART);
    DisplayEnable();
    DisplaySetData();
    while (counter--) {
#if (COLOR_DEPTH == 16)
        DeviceWrite(_color);
#elif (COLOR_DEPTH == 24)
        DeviceWrite(_color >> 8);
        DeviceWrite((_color << 8) | (_color >> 16));
        DeviceWrite(_color);
#endif
    }

    DisplayDisable();

    return 1;
}

/*********************************************************************
 * Function: void ClearDevice(void)
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: clears screen with current color
 *
 * Note: none
 *
 ********************************************************************/
void ClearDevice(void) {
    /*
 DWORD counter;
 DWORD  xcounter, ycounter;

 SetArea(0,0,DISP_HOR_RESOLUTION-1,DISP_VER_RESOLUTION-1);

 WriteCommand(CMD_WR_MEMSTART);

 DisplayEnable();
 for(ycounter=0;ycounter<DISP_VER_RESOLUTION;ycounter++)
 {
  for(xcounter=0;xcounter<DISP_HOR_RESOLUTION;xcounter++)
  {
   WriteData(_color);
  }
 }

 DisplayDisable();
     */
    Bar(0, 0, GetMaxX(), GetMaxY());
}

/*********************************************************************
 * Function: IsDeviceBusy()
 *
 * Overview: Returns non-zero if LCD controller is busy
 *           (previous drawing operation is not completed).
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: Busy status.
 *
 * Side Effects: none
 *
 ********************************************************************/
WORD IsDeviceBusy(void) {
    //Check the wait pin
    return 0; //(DisplayWaitPin() == 0);
}

#ifdef USE_TRANSPARENT_COLOR

/*********************************************************************
 * Function:  void TransparentColorEnable(GFX_COLOR color)
 *
 * Overview: Sets current transparent color.
 *
 * PreCondition: none
 *
 * Input: color - Color value chosen.
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TransparentColorEnable(GFX_COLOR color) {
    _colorTransparent = color;
    _colorTransparentEnable = TRANSPARENT_COLOR_ENABLE;

}
#endif



