/*****************************************************************************
 *  Module for Microchip Graphics Library
 *  Ilitek RA8872 LCD controller driver
 *****************************************************************************
 * FileName:        ILI9163.c
 * Processor:       PIC32
 * Compiler:        MPLAB C32
 * Company:         Displaytech Ltd.
 *
 * Date        	Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 06/22/12     Initial Revision
  ***************************************************************************/

#include "HardwareProfile.h"
#include "RA8872.h"
#include "TimeDelay.h"

#if defined(GFX_USE_DISPLAY_CONTROLLER_RA8872)
//Error checking
#ifndef USE_8BIT_PMP
#error("The RA8872 is only compatible with 8 bit PMP");
#endif
#ifdef USE_PALETTE
#error("The RA8872 driver does not support a color palette")
#endif

#include "Compiler.h"
#include "TimeDelay.h"
#include "Graphics/DisplayDriver.h"
#include "RA8872.h"
#include "Graphics/Primitive.h"
#if defined (USE_GFX_PMP)
#include "Graphics/gfxpmp.h"
#elif defined (USE_GFX_EPMP)
#include "Graphics/gfxepmp.h"
#endif    

// Color
GFX_COLOR _color;
#ifdef USE_TRANSPARENT_COLOR
GFX_COLOR _colorTransparent;
SHORT _colorTransparentEnable;
#endif

// Clipping region control
SHORT _clipRgn;

// Clipping region borders
SHORT _clipLeft;
SHORT _clipTop;
SHORT _clipRight;
SHORT _clipBottom;

//Usage semaphore
BYTE _inUse = 0;
#define SET_DEVICE_BUSY() (_inUse = 1)
#define SET_DEVICE_READY() (_inUse = 0)
#define DEVICE_IN_USE() (_inUse == 1)


/////////////////////// LOCAL FUNCTIONS PROTOTYPES ////////////////////////////

static inline void WriteCommand(BYTE cmd);
static inline void WriteData(BYTE cmd);
static inline BYTE ReadData(void);
static inline BYTE ReadStatus(void);
static inline BYTE ReadRegister(BYTE reg);
static inline void WriteRegister(BYTE reg, BYTE dat);
static inline void ActiveWindow(WORD XL, WORD XR, WORD YT, WORD YB);
static inline void XYCoordinate(WORD X, WORD Y);

/////////////////////// LOCAL FUNCTIONS ////////////////////////////

/*********************************************************************
 * Function:  static inline void WriteCommand(BYTE cmd)
 *
 * PreCondition: none
 *
 * Input: Command BYTE to write
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Writes an 8-bit command value to the RA8872
 *
 * Note: none
 *
 ********************************************************************/
static inline void WriteCommand(BYTE cmd)
{
    DisplayEnable();        //Set the CS Low
    DisplaySetCommand();    //Set the RS high
    DeviceWrite(cmd);       //Send PMP data
    DisplayDisable();       //Set the CS High
}

/*********************************************************************
 * Function:  static inline void WriteData(BYTE cmd)
 *
 * PreCondition: WriteCommand() should have been called previously in
 *               order to specify the type of data which is being written
 *
 * Input: Data BYTE to write
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Writes an 8-bit data value to the RA8872
 *
 * Note: none
 *
 ********************************************************************/
static inline void WriteData(BYTE cmd)
{
    DisplayEnable();        //Set the CS Low
    DisplaySetData();       //Set the RS low
    DeviceWrite(cmd);       //Dump data on the PMP
    DisplayDisable();       //Set the CS High
}

/*********************************************************************
 * Function:  static inline BYTE ReadData(void)
 *
 * PreCondition: WriteCommand() should have been called previously in
 *               order to specify the type of data which is being read
 *
 * Input: none
 *
 * Output: BYTE of data which is to be read
 *
 * Side Effects: none
 *
 * Overview: Reads an 8-bit data value from the RA8872
 *
 * Note: none
 *
 ********************************************************************/
static inline BYTE ReadData(void)
{
    BYTE data;

    DisplayEnable();        //Set the CS low
    DisplaySetData();       //Set the RS low
    data = DeviceRead();    //Read one byte of data
    DisplayDisable();       //Set the CS high

    return data;
}

/*********************************************************************
 * Function:  static inline BYTE ReadStatus(void)
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: The status register BYTE representation
 *
 * Side Effects: none
 *
 * Overview: Reads the 8-bit status register
 *
 * Note: none
 *
 ********************************************************************/
static inline BYTE ReadStatus(void)
{
    BYTE status;

    DisplayEnable();            //Set the CS low
    DisplaySetCommand();        //Set the RS high
    status = DeviceRead();      //Read one byte of data
    DisplayDisable();           //Set the CS high

    return status;
}

/*********************************************************************
 * Function:  static inline BYTE ReadRegister(reg)
 *
 * PreCondition: none
 *
 * Input: The register address to be returned
 *
 * Output: The register value
 *
 * Side Effects: none
 *
 * Overview: Reads an 8-bit register value from the RA8872
 *
 * Note: This function only reads 8-bit registers.  Longer registers
 *       should be read manually.
 *
 ********************************************************************/
static inline BYTE ReadRegister(BYTE reg)
{
    BYTE dat;

    WriteCommand(reg);
    dat = ReadData();
}

/*********************************************************************
 * Function:  static inline void WriteRegister(BYTE reg, BYTE dat)
 *
 * PreCondition: none
 *
 * Input: The register to be written and the value to write
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Writes an 8-bit register value to the RA8872
 *
 * Note: This function only writes 8-bit registers.  Longer registers
 *       should be written manually.
 *
 ********************************************************************/
static inline void WriteRegister(BYTE reg, BYTE dat)
{
    WriteCommand(reg);
    WriteData(dat);
}

/*********************************************************************
 * Function:  static inline void ActiveWindow(WORD XL, WORD XR, WORD YT, WORD YB)
 *
 * PreCondition: none
 *
 * Input:       XL: The leftmost position of the window
 *              XR: The rightmost position of the window
 *              YT: The topmost position of the window
 *              YB: The bottommost position of the window
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Sets the active window, the area of the frame buffer which
 *           can be written to and will display on the screen.
 *
 * Note: none
 *
 ********************************************************************/
static inline void ActiveWindow(WORD XL, WORD XR, WORD YT, WORD YB)
{
    WORD swap;

    //Some error checking
    if(XL > XR)
    {
        swap = XL;
        XL = XR;
        XR = swap;
    }
    else if (YT > YB)
    {
        swap = YT;
        YT = YB;
        YB = swap;
    }

    while(IsDeviceBusy());

    SET_DEVICE_BUSY();
    
    //setting active window X
    WriteRegister(HSAW0,(BYTE)(XL));
    WriteRegister(HSAW1,(BYTE)(XL >> 8) & 0x03);
    WriteRegister(HEAW0,(BYTE)(XR));
    WriteRegister(HEAW1,(BYTE)(XR >> 8) & 0x03);

    //setting active window Y
    WriteRegister(VSAW0,(BYTE)(YT));
    WriteRegister(VSAW1,(BYTE)(YT >> 8) & 0x01);
    WriteRegister(VEAW0,(BYTE)(YB));
    WriteRegister(VEAW1,(BYTE)(YB >> 8) & 0x01);

    SET_DEVICE_READY();
}

/*********************************************************************
 * Function:  static inline void XYCoordinate(WORD X, WORD Y)
 *
 * PreCondition: none
 *
 * Input:       X: The X position of the memory cursor set
 *              Y: The Y position of the memory cursor set
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Sets the position of the memory write cursor.  This function
 *           should be followed by a memory write command and memory data writes.
 *
 * Note: none
 *
 ********************************************************************/
static inline void XYCoordinate(WORD X, WORD Y)
{
    while(IsDeviceBusy());

    SET_DEVICE_BUSY();

    //Set the X coordinate of the memory cursor
    WriteRegister(CURH0, (BYTE)X);
    WriteRegister(CURH1, (BYTE)(X>>8));

    //Set the Y coordinate of the memory cursor
    WriteRegister(CURV0, (BYTE)Y);
    WriteRegister(CURV1, (BYTE)(Y>>8));

    SET_DEVICE_READY();
}

/////////////////////// GLOBAL FUNCTIONS ////////////////////////////

/*********************************************************************
 * Function:  void ResetDevice()
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Sets up the LCD comm pins, initializes PMP, resets LCD
 *
 * Note: none
 *
 ********************************************************************/
void ResetDevice(void)
{
    //Set all I/O pins as inputs, reset the device to cause auto-initialization
    DisplayCSSetInput();
    DisplayRSSetInput();
    DisplayWRSetInput();
    DisplayRDSetInput();
    DisplayINTSetInput();
    DisplayWAITSetInput();
    DisplayDATASetInput();

    //Force a reset
    DisplayResetEnable();
    DisplayResetConfig();
    DelayMs(100);
    DisplayResetDisable();
    DelayMs(3000); //1.5 second delay for auto-initialize to finish.

    /////////////////////////////////////////////////////////////////////
    // Initialize the output pins and parallel master port
    /////////////////////////////////////////////////////////////////////
    DriverInterfaceInit();

    /////////////////////////////////////////////////////////////////////
    // Initialize the device
    /////////////////////////////////////////////////////////////////////

    //Set the device for 16-bit data width (5-6-5)
    WriteRegister(SYSR,0x08);

    //Set the horizontal and vertical resolutions
    WriteRegister(HDWR,(BYTE)(DISP_HOR_RESOLUTION/8-1));
    WriteRegister(VDHR0,(BYTE)(DISP_VER_RESOLUTION-1));
    WriteRegister(VDHR1,(BYTE)((DISP_VER_RESOLUTION-1)>>8));

    //Set rotations and horizontal write directions for
    //different displays
#if (DISP_ORIENTATION == 0)
#if (DISP_HOR_RESOLUTION == 160)
    WriteRegister(DPCR,ROT_270_MASK|HDIR_MASK);
#elif(DISP_HOR_RESOLUTION == 320)
    WriteRegister(DPCR,HDIR_MASK|ROT_90_MASK);
#else
    //WriteRegister(DPCR,ROT_270_MASK);
#endif
#elif (DISP_ORIENTATION == 90)
#if (DISP_HOR_RESOLUTION == 320)
    WriteRegister(DPCR,HDIR_MASK|ROT_90_MASK);
    //WriteRegister(DPCR,ROT_90_MASK);
#endif
#endif

    //Turn on the display
    WriteRegister(PWRR,0x80);

    /////////////////////////////////////////////////////////////////////
    // Clear the display buffer with all zeros so the display will not
    // show garbage data when initially powered up
    /////////////////////////////////////////////////////////////////////
    SetColor(BLACK);
    ClearDevice();
}

/*********************************************************************
 * Function: void PutPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: puts pixel
 *
 * Note: none
 *
 ********************************************************************/
void PutPixel(SHORT x, SHORT y)
{
    while (IsDeviceBusy() != 0);

    if (_clipRgn)
    {
        if (x < _clipLeft)
            return;
        if (x > _clipRight)
            return;
        if (y < _clipTop)
            return;
        if (y > _clipBottom)
            return;
    }

#ifdef USE_TRANSPARENT_COLOR
    if(_colorTransparentEnable == TRANSPARENT_COLOR_ENABLE){
        if(_colorTransparent == _color){
            return;
        }
    }
#endif

    SET_DEVICE_BUSY();

    //Select the pixel
#if (DISP_ORIENTATION == 0)
    XYCoordinate(x,y);
#elif (DISP_ORIENTATION == 90)
    XYCoordinate(y,x);
#endif
    WriteCommand(MRWC);
    WriteData((BYTE)(_color>>8));
    WriteData((BYTE)_color);

    SET_DEVICE_READY();
}

/*********************************************************************
 * Function: WORD GetPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: pixel color
 *
 * Side Effects: none
 *
 * Overview: returns pixel color at x,y position
 *
 * Note: none
 *
 ********************************************************************/
WORD GetPixel(SHORT y, SHORT x)
{
    GFX_COLOR value;

#ifndef USE_NONBLOCKING_CONFIG
    while (IsDeviceBusy() != 0);
    /* Ready */
#else
    if (IsDeviceBusy() != 0)
        return (0);
#endif

    SET_DEVICE_BUSY();

    //Select the pixel
#if (DISP_ORIENTATION == 0)
    WriteRegister(RCURH0, (BYTE)x);
    WriteRegister(RCURH01, (BYTE)(x >> 8));
    WriteRegister(RCURV0, (BYTE)y);
    WriteRegister(RCURV1, (BYTE)(y >> 8));
#elif (DISP_ORIENTATION == 90)
    WriteRegister(RCURH0, (BYTE)y);
    WriteRegister(RCURH01, (BYTE)(y >> 8));
    WriteRegister(RCURV0, (BYTE)x);
    WriteRegister(RCURV1, (BYTE)(x >> 8));
#endif
    
    //Read the memory
    WriteCommand(MRWC);
    value = ReadData();
    value <<= 8;
    while(IsDeviceBusy()){}
    value |= ReadData();

    SET_DEVICE_READY();

    return (value);
}

/*********************************************************************
 * Function: void GetPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *desArray, WORD len, WORD dir)
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data read
 *              yStart: The y starting position of the data read
 *              *desArray: The array of data to be read
 *              len: The length of the data to read
 *              dir: The direction of the read (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Reads an array of data of length len from the frame buffer
 *           on the RA8872.  This function is used for alphablending.
 *
 * Note: none
 *
 ********************************************************************/
void GetPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *desArray, WORD len, WORD dir)
{
    while (IsDeviceBusy() != 0);

    SET_DEVICE_BUSY();

    //Set the starting position of the read cursor
#if (DISP_ORIENTATION == 0)
    WriteRegister(RCURH0, (BYTE)xStart);
    WriteRegister(RCURH01, (BYTE)(xStart>>8));
    WriteRegister(RCURV0, (BYTE)yStart);
    WriteRegister(RCURV1, (BYTE)(yStart>>8));
#elif (DISP_ORIENTATION == 90)
    WriteRegister(RCURH0, (BYTE)(yStart));
    WriteRegister(RCURH01, (BYTE)(yStart>>8));
    WriteRegister(RCURV0, (BYTE)(xStart));
    WriteRegister(RCURV1, (BYTE)(xStart>>8));
    WriteRegister(MCRD,0x02);   //Change the read direction
#endif

    //Start the memory write
    WriteCommand(MRWC);

    DisplayEnable();    //Set the CS low
    DisplaySetData();   //Set the RS low
    while(len--)
    {
        while(IsDeviceBusy());  //Wait for the device to be ready
        *desArray = DeviceRead();   //Read the top byte
        *desArray <<= 8;            //Shift the byte
        *desArray |= DeviceRead();  //Read the low byte
        desArray++;                 //Increment the pointer
    }
    DisplayDisable();   //Set the CS high

    SET_DEVICE_READY();
}

/*********************************************************************
 * Function: void PutPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *srcArray, WORD len, WORD dir)
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data write
 *              yStart: The y starting position of the data write
 *              *desArray: The array of data to be written
 *              len: The length of the data to write
 *              dir: The direction of the write (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Writes an array of data of length len to the frame buffer
 *           on the RA8872.  This function is used for alphablending
 *
 * Note: none
 *
 ********************************************************************/
void PutPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *srcArray, WORD len, WORD dir)
{
#ifdef USE_TRANSPARENT_COLOR
    SHORT x, y;
#endif

    while (IsDeviceBusy() != 0);

    SET_DEVICE_BUSY();
    
    if (_clipRgn)
    {
        if (xStart < _clipLeft)
            return;
        if (xStart > _clipRight)
            return;
        if (yStart < _clipTop)
            return;
        if (yStart > _clipBottom)
            return;
    }

    //Set the starting position of the write cursor
    #if (DISP_ORIENTATION == 0)
    XYCoordinate(xStart,yStart);
    #elif (DISP_ORIENTATION == 90)
    WriteRegister(MWCR0,0x08);
    XYCoordinate(yStart,xStart);
    #endif

    //Start the data write
    WriteCommand(MRWC);

    DisplayEnable();        //Set the CS low
    DisplaySetData();       //Set the RS low
    while(len--)
    {
#ifdef USE_TRANSPARENT_COLOR
        if(_colorTransparentEnable == TRANSPARENT_COLOR_ENABLE){
            if(_colorTransparent == *srcArray){
                DisplayDisable();
                
                x = ReadRegister(CURH1);
                x <<= 8;
                x = ReadRegister(CURH0);
                y = ReadRegister(CURV1);
                y<<=8;
                y = ReadRegister(CURV0);
#if (DISP_ORIENTATION == 0)
                x+=1;
                WriteRegister(CURH0,x);
                WriteRegister(CURH1,x<<8);
#elif (DISP_ORIENTATION == 90)
                y+=1;
                WriteRegister(CURV0,y);
                WriteRegister(CURV1,y<<8);
#endif
                WriteCommand(MRWC);
                DisplayEnable();
                DisplaySetData();
                ++srcArray;
                continue;
            }
        }
#endif
        DeviceWrite((BYTE)(*srcArray>>8));   //Write the top data byte
        DeviceWrite((BYTE)(*srcArray));     //Write the bottom data byte
        ++srcArray;                         //Increment the source array
    }

    DisplayDisable();       //Set the CS high

    SET_DEVICE_READY();
}

/*********************************************************************
 * Function: inline void ScrollWindow(WORD XL, WORD XR, WORD YT, WORD YB)
 *
 * PreCondition: none
 *
 * Input:       XL: The leftmost position of the scroll window
 *              XR: The rightmost position of the scroll window
 *              YT: The topmost position of the scroll window
 *              YB: The bottommost position of the scroll window
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Sets the window for horizontal or vertical scrolling on the screen
 *
 * Note: none
 *
 ********************************************************************/
inline void ScrollWindow(WORD XL, WORD XR, WORD YT, WORD YB)
{
    while (IsDeviceBusy() != 0);

    SET_DEVICE_BUSY();
    
    //If the screen is rotated, swap X and Y
#if (DISP_ORIENTATION == 0)
    //setting scroll window X
    WriteRegister(HSSW0,(BYTE)(XL));
    WriteRegister(HSSW1,(BYTE)(XL >> 8) & 0x03);
    WriteRegister(HESW0,(BYTE)(XR));
    WriteRegister(HESW1,(BYTE)(XR >> 8) & 0x03);

    //setting scroll window Y
    WriteRegister(VSSW0,(BYTE)(YT));
    WriteRegister(VSSW1,(BYTE)(YT >> 8) & 0x01);
    WriteRegister(VESW0,(BYTE)(YB));
    WriteRegister(VESW1,(BYTE)(YB >> 8) & 0x01);
#endif
    #if (DISP_ORIENTATION == 90)
//setting scroll window X
    WriteRegister(HSSW0,(BYTE)(YT));
    WriteRegister(HSSW1,(BYTE)(YT >> 8) & 0x03);
    WriteRegister(HESW0,(BYTE)(YB));
    WriteRegister(HESW1,(BYTE)(YB >> 8) & 0x03);

    //setting scroll window Y
    WriteRegister(VSSW0,(BYTE)(XL));
    WriteRegister(VSSW1,(BYTE)(XL >> 8) & 0x01);
    WriteRegister(VESW0,(BYTE)(XR));
    WriteRegister(VESW1,(BYTE)(XR >> 8) & 0x01);
#endif

    SET_DEVICE_READY();
}

/*********************************************************************
 * Function: inline void VerticalScroll(WORD offset)
 *
 * PreCondition: The scroll window must already be set
 *
 * Input:       offset: The pixel amount to shift the scroll window vertically by
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Scrolls the scroll window vertically by offset pixels
 *
 * Note: none
 *
 ********************************************************************/
inline void VerticalScroll(WORD offset)
{
    while (IsDeviceBusy() != 0);

    SET_DEVICE_BUSY();

    //Swap the horiz/vert if the screen is rotated.
#if (DISP_ORIENTATION == 0)
    WriteRegister(VOFS0,(BYTE)offset);
    WriteRegister(VOFS1,(BYTE)((offset >> 8) & 0x01));
#elif (DISP_ORIENTATION == 90)
    WriteRegister(HOFS0,(BYTE)offset);
    WriteRegister(HOFS1,(BYTE)((offset >> 8) & 0x03));
#endif

    SET_DEVICE_READY();
}

/*********************************************************************
 * Function: inline void HorizontalScroll(WORD offset)
 *
 * PreCondition: The scroll window must already be set
 *
 * Input:       offset: The pixel amount to shift the scroll window horizontally by
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Scrolls the scroll window horizontally by offset pixels
 *
 * Note: none
 *
 ********************************************************************/
inline void HorizontalScroll(WORD offset)
{
    while (IsDeviceBusy() != 0);

    SET_DEVICE_BUSY();

    //Swap the horiz/vert if the screen is rotated.
#if (DISP_ORIENTATION == 0)
    WriteRegister(HOFS0,(BYTE)offset);
    WriteRegister(HOFS1,(BYTE)((offset >> 8) & 0x03));
#elif (DISP_ORIENTATION == 90)
    WriteRegister(VOFS0,(BYTE)offset);
    WriteRegister(VOFS1,(BYTE)((offset >> 8) & 0x01));
#endif

    SET_DEVICE_READY();
}

/*********************************************************************
 * Function: WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom)
 *
 * PreCondition: none
 *
 * Input: left,top - top left corner coordinates,
 *        right,bottom - bottom right corner coordinates
 *
 * Output: For NON-Blocking configuration:
 *         - Returns 0 when device is busy and the shape is not yet completely drawn.
 *         - Returns 1 when the shape is completely drawn.
 *         For Blocking configuration:
 *         - Always return 1.
 *
 * Side Effects: none
 *
 * Overview: draws rectangle filled with current color
 *
 * Note: none
 *
 ********************************************************************/
WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom)
{

#ifndef USE_NONBLOCKING_CONFIG
    while (IsDeviceBusy() != 0);
    /* Ready */
#else
    if (IsDeviceBusy() != 0)
        return (0);
#endif

    if(top > bottom || left > right)
        return 1;

    SET_DEVICE_BUSY();

    if (_clipRgn)
    {
        if (left < _clipLeft)
            left = _clipLeft;
        if (right > _clipRight)
            right = _clipRight;
        if (top < _clipTop)
            top = _clipTop;
        if (bottom > _clipBottom)
            bottom = _clipBottom;
    }

    //If this is only one pixel, call the PutPixel() function
    if(left == right && top == bottom)
        PutPixel(left,right);
    else
    {
        //Set the active window
#if (DISP_ORIENTATION == 0)
        ActiveWindow(left,right,top,bottom);
#elif (DISP_ORIENTATION == 90)
        ActiveWindow(top,bottom,left,right);
#endif

        //Set the BTE background
        WriteRegister(BGCR0,(BYTE)(_color>>11));//Red
        WriteRegister(BGCR1,(BYTE)(_color>>5)); //Green
        WriteRegister(BGCR2,(BYTE)(_color));    //Blue

        //Clear to the BTE background
        WriteRegister(MCLR,0xC0);
    }

    SET_DEVICE_READY();

    //Wait for the BTE to complete
    while(IsDeviceBusy());

    SET_DEVICE_BUSY();
    
    //Reset the active window
#if (DISP_ORIENTATION == 0)
       ActiveWindow(0,GetMaxX(),0,GetMaxY());
#elif (DISP_ORIENTATION == 90)
       ActiveWindow(0,GetMaxY(),0,GetMaxX());
#endif

    SET_DEVICE_READY();

    return (1);
}

/*********************************************************************
 * Function: void ClearDevice(void)
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: clears screen with current color
 *
 * Note: none
 *
 ********************************************************************/
void ClearDevice(void) {
#ifndef USE_NONBLOCKING_CONFIG
    Bar(0, 0, GetMaxX(), GetMaxY());
#else
    while(!Bar(0, 0, GetMaxX(), GetMaxY()));
#endif
    
}

/*********************************************************************
 * Function: SetClipRgn(left, top, right, bottom)
 *
 * Overview: Sets clipping region.
 *
 * PreCondition: none
 *
 * Input: left - Defines the left clipping region border.
 *		 top - Defines the top clipping region border.
 *		 right - Defines the right clipping region border.
 *	     bottom - Defines the bottom clipping region border.
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void SetClipRgn(SHORT left, SHORT top, SHORT right, SHORT bottom) {
    _clipLeft = left;
    _clipTop = top;
    _clipRight = right;
    _clipBottom = bottom;

}

/*********************************************************************
 * Function: SetClip(control)
 *
 * Overview: Enables/disables clipping.
 *
 * PreCondition: none
 *
 * Input: control - Enables or disables the clipping.
 *			- 0: Disable clipping
 *			- 1: Enable clipping
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void SetClip(BYTE control) {
    _clipRgn = control;
}

#ifdef USE_TRANSPARENT_COLOR
/*********************************************************************
 * Function:  void TransparentColorEnable(GFX_COLOR color)
 *
 * Overview: Sets current transparent color.
 *
 * PreCondition: none
 *
 * Input: color - Color value chosen.
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TransparentColorEnable(GFX_COLOR color) {
    _colorTransparent = color;
    _colorTransparentEnable = TRANSPARENT_COLOR_ENABLE;

}
#endif

/*********************************************************************
 * Function: IsDeviceBusy()
 *
 * Overview: Returns non-zero if LCD controller is busy
 *           (previous drawing operation is not completed).
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: Busy status.
 *
 * Side Effects: none
 *
 ********************************************************************/
WORD IsDeviceBusy(void) {
    //Check the wait pin
    return (DisplayWaitPin() == 0);
}

#ifdef USE_TOUCHSCREEN

SHORT xRaw,yRaw;                        //raw x and y measurements
const WORD mchpTouchScreenVersion = 1;  //Touchscreen version for Microchip
#define MAX_JUMP (10*4096/DISP_HOR_RESOLUTION) //Noise rejection threshold

/*********************************************************************
 * Function: SHORT TouchDetectPosition(void)
 *
 * Overview: Read the x and y positions of the touchscreen controller
 *           into xRaw and yRaw.
 *
 * PreCondition: The touch hardware needs to have been initialized prior.
 *
 * Input: none
 *
 * Output: The status
 *
 * Side Effects: none
 *
 ********************************************************************/
SHORT TouchDetectPosition(void)
{
    BYTE xData,yData,lowData;
    BYTE status = 0;
    static SHORT lastX = 0, lastY = 0;

    while(IsDeviceBusy());
        
    if(DEVICE_IN_USE())
        return 0;

    status = ReadRegister(INTC);

    if(!(status&0x40))
    {
        WriteRegister(INTC,0x40);
        WriteRegister(INTC,0x44);
        return 0;
    }

    if(status & 0x04)
    {
        while(IsDeviceBusy());

        xData = ReadRegister(TPXH);
        yData = ReadRegister(TPYH);
        lowData = ReadRegister(TPXYL);

        WriteRegister(INTC,status);

#if (DISP_ORIENTATION == 90)
        xRaw = (xData << 2) | (lowData & 0x03);
        yRaw = (yData << 2) | (lowData & 0x0C);

        if(yRaw > 500 && yRaw < 550)
        {
            xRaw = -1;
            yRaw = -1;
        }
#else
        xRaw = (yData << 2) | (lowData & 0x0C);
        yRaw = (xData << 2) | (lowData & 0x03);
#endif
        //Error checking
        if((lastX != -1 && lastY != -1) &&
           (abs((SHORT)xRaw - lastX) > MAX_JUMP ||
            abs((SHORT)yRaw - lastY) > MAX_JUMP))
        {
            xRaw = -1;
            yRaw = -1;
        }
    }
    else
    {
        xRaw = -1;
        yRaw = -1;
    }

    lastX = xRaw;
    lastY = yRaw;

    return 1;
}

/*********************************************************************
 * Function: void TouchHardwareInit(void *initValues)
 *
 * Overview: Initialize the RA8872 touchscreen controller
 *
 * PreCondition: The RA8872 should have been initialized prior to this
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TouchHardwareInit(void *initValues)
{
    BYTE temp;
    //Set up the RA8872 to auto mode, polling
    WriteRegister(TPCR0,0b11110000);
    WriteRegister(TPCR1,0x84);

    //Turn on the TP interrupt
//    temp = ReadRegister(INTC);
//    temp |= 0x40;
    WriteRegister(INTC,0x40);

    //Immediately clear the interrupt
    temp = ReadRegister(INTC);
    temp |= 0x04;
    WriteRegister(INTC,temp);
}

/*********************************************************************
 * Function: SHORT TouchGetX(void)
 *
 * Overview: Returns the converted x position
 *
 * PreCondition: The touch hardware needs to have been initialized prior.
 *
 * Input: none
 *
 * Output: The converted x position in pixels
 *
 * Side Effects: none
 *
 ********************************************************************/
SHORT TouchGetX(void)
{
    if(xRaw == -1)
        return -1;
    else
#if (DISP_ORIENTATION == 0)
        return (SHORT)(DISP_HOR_RESOLUTION*xRaw/1024);
#elif (DISP_ORIENTATION == 90)
        return (SHORT)(DISP_VER_RESOLUTION*xRaw/1024);
#endif
}

/*********************************************************************
 * Function: SHORT TouchGetRawX(void)
 *
 * Overview: Returns the raw x position
 *
 * PreCondition: The touch hardware needs to have been initialized prior.
 *
 * Input: none
 *
 * Output: The raw x position in ADC counts
 *
 * Side Effects: none
 *
 ********************************************************************/
SHORT TouchGetRawX(void)
{
    return (SHORT)xRaw;
}

/*********************************************************************
 * Function: SHORT TouchGetY(void)
 *
 * Overview: Returns the converted y position
 *
 * PreCondition: The touch hardware needs to have been initialized prior.
 *
 * Input: none
 *
 * Output: The converted y position in pixels
 *
 * Side Effects: none
 *
 ********************************************************************/
SHORT TouchGetY(void)
{
    if(yRaw == -1)
        return -1;
    else
#if (DISP_ORIENTATION == 0)
        return (SHORT)(DISP_VER_RESOLUTION*yRaw/1024);
#elif (DISP_ORIENTATION == 90)
        return (SHORT)(DISP_HOR_RESOLUTION*(1024-yRaw)/1024);
#endif
}

/*********************************************************************
 * Function: SHORT TouchGetRawY(void)
 *
 * Overview: Returns the raw y position
 *
 * PreCondition: The touch hardware needs to have been initialized prior.
 *
 * Input: none
 *
 * Output: The raw y position in ADC counts
 *
 * Side Effects: none
 *
 ********************************************************************/
SHORT TouchGetRawY(void)
{
    return (SHORT)yRaw;
}

/*********************************************************************
 * Function: void TouchStoreCalibration(void)
 *
 * Overview: Does nothing
 *
 * PreCondition: None
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TouchStoreCalibration(void)
{
    //There is no calibration data to store
}

/*********************************************************************
 * Function: void TouchLoadCalibration(void)
 *
 * Overview: Does nothing
 *
 * PreCondition: None
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TouchLoadCalibration(void)
{
    // Calibration data is loaded by the controller
}

/*********************************************************************
 * Function: void TouchCalHWGetPoints(void)
 *
 * Overview: Does nothing
 *
 * PreCondition: None
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TouchCalHWGetPoints(void)
{
//    SHORT x, y;
//
//    SetColor(RED);
//    while(1)
//    {
//        x = TouchGetX();
//        y = TouchGetY();
//
//        if(x!=-1 && y!=-1)
//        {
//            Bar(x-1,y-1,x+1,y+1);
//        }
//    }
    //There are no calibrations to work with
}

#endif // #if USE_TOUCHSCREEN
#endif // #if (GFX_USE_DISPLAY_CONTROLLER_SSD1926) 
