/*****************************************************************************
 *
 * This document contains proprietary information and except with
 * written permission of Displaytech, Ltd., such information shall
 * not be published or disclosed to others or used for any purpose
 * other than for the operation and maintenance of the equipment
 * and software with which it was procured, and the document shall
 * not be copied in whole or in part.
 *
 * Copyright 2008-2015 Displaytech, Ltd. All Rights Reserved.
 *
 *****************************************************************************
 *  Module for Microchip Graphics Library
 *  Ilitek ILI9341 LCD controller driver
 *****************************************************************************
 * FileName:        ILI9341.c
 * Processor:       PIC32
 * Compiler:           MPLAB C32
 * Company:         Displaytech Ltd.
 *
 * Date            Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 06/22/12     Initial Revision
 *************/
#include "HardwareProfile.h"

#if defined(GFX_USE_DISPLAY_CONTROLLER_ILI9341)

#include "Compiler.h"
#include "TimeDelay.h"
#include "Graphics/DisplayDriver.h"
#include "ILI9341.h"
#include "Graphics/gfxtcon.h"
#include "Graphics/Primitive.h"


#if defined (USE_GFX_PMP)
#include "Graphics/gfxpmp.h"
#elif defined (USE_GFX_EPMP)
#include "Graphics/gfxepmp.h"
#endif

// Color
GFX_COLOR _color;
#ifdef USE_TRANSPARENT_COLOR
GFX_COLOR _colorTransparent;
SHORT _colorTransparentEnable;
#endif

// Clipping region control
SHORT _clipRgn;

// Clipping region borders
SHORT _clipLeft;
SHORT _clipTop;
SHORT _clipRight;
SHORT _clipBottom;

#define RED8(color16)   (BYTE) ((color16 & 0xF800) >> 8)
#define GREEN8(color16) (BYTE) ((color16 & 0x07E0) >> 3)
#define BLUE8(color16)  (BYTE) ((color16 & 0x001F) << 3)

/////////////////////// LOCAL FUNCTIONS PROTOTYPES ////////////////////////////
void WriteCommand(BYTE command);
void WriteData(WORD data);
void SetPageRange(WORD start, WORD end);
void SetColumnRange(WORD start, WORD end);
void GetPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *desArray, WORD len, WORD dir);

#ifdef USE_PALETTE
extern void *_palette;
static BYTE PaletteBpp = 16;
extern BYTE blPaletteChangeError;
extern void *pPendingPalette;
extern WORD PendingStartEntry;
extern WORD PendingLength;
#endif

//Set the column memory range

void SetColumnRange(WORD start, WORD end) {
    WriteCommand(COLUMN_ADDRESS_SET); //Set Column Address
    WriteData((BYTE) (start >> 8)); //Start address [15:8]
    WriteData((BYTE) start); //Start address [7:0]
    WriteData((BYTE) (end >> 8)); //End address [15:8]
    WriteData((BYTE) end); //End address [7:0]
}

//Set the page memory range

void SetPageRange(WORD start, WORD end) {
    WriteCommand(PAGE_ADDRESS_SET); //Set Column Address
    WriteData((BYTE) (start >> 8)); //Start address [15:8]
    WriteData((BYTE) start); //Start address [7:0]
    WriteData((BYTE) (end >> 8)); //End address [15:8]
    WriteData((BYTE) end); //End address [7:0]
}

//Write a command

void WriteCommand(BYTE command) {
    //Set the CS Low
    DisplayEnable();

    //Set the RS pin LOW
    DisplaySetCommand();

    //Dump the command byte into the PMP
    DeviceWrite(command);
    //PMDIN1 = (WORD) command;

    //Wait for the transmission end
    PMPWaitBusy();

    //Set the RS high, next bytes will likely be data
    DisplaySetData();

    //Set the CS high
    DisplayDisable();
}

//Write 1 word of data (after command)

void WriteData(WORD data) {
    //Set the CS low
    DisplayEnable();

    //Set the RS pin high
    DisplaySetData();

    //Dump the data into the PMP
    DeviceWrite(data);
    //PMDIN1 = data;

    //Wait for the transmission end
    PMPWaitBusy();

    //Set the CS high
    DisplayDisable();
}

WORD ReadData(void) {
    WORD data;

    //Set the CS low
    DisplayEnable();

    //Dump the data into the PMP
    data = DeviceRead();

    //Set the CS high
    DisplayDisable();

    return data;
}

WORD ReadCommand(BYTE cmd) {
    WORD dat;

    DisplayEnable();
    DisplaySetCommand();
    DeviceWrite(cmd);
    //PMDIN1 = cmd;
    PMPWaitBusy();
    DisplaySetData();
    dat = DeviceRead();
    dat = DeviceRead();
    DisplayDisable();
    return dat;
}

/*********************************************************************
 * Function:  void ResetDevice()
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: resets LCD, initializes PMP
 *
 * Note: none
 *
 ********************************************************************/
void ResetDevice(void) {
    /////////////////////////////////////////////////////////////////////
    // Initialize the device
    /////////////////////////////////////////////////////////////////////
    DriverInterfaceInit();

    DisplayResetEnable();
    DisplayResetConfig();
    DelayMs(10);
    DisplayResetDisable();
    DelayMs(120);

    DisplayBacklightOn();
    DisplayBacklightConfig();

    //Software Reset
    WriteCommand(SOFTWARE_RESET);
    //Supposed to wait at least 120ms, wait 250 to be safe
    DelayMs(250);
    WriteCommand(DISPLAY_OFF); // display off

    //Power
    WriteCommand(POWER_1); //power control
    WriteData(0x26);
    WriteCommand(POWER_2); //power control
    WriteData(0x11);
    WriteCommand(VCOM_1); //vcom control
    WriteData(0x5c); //35
    WriteData(0x4c); //3E
    WriteCommand(VCOM_2); //vcom control
    WriteData(0x94);

    //Pixel format
    WriteCommand(PIXEL_FORMAT_SET);
    WriteData(0x65); //16-bit 5-6-5 MCU

    //Memory access
    WriteCommand(MEMORY_ACCESS_CTRL);
#if (DISP_HOR_RESOLUTION == 320) //320x240
    WriteData(0x00E8); //Swap row/columns, RGB/BGR swap
#else //240x320
    WriteData(0x0048); //Swap row/columns, RGB/BGR swap
#endif

    //Frame Rate
    WriteCommand(FRAME_RATE_NORMAL); // frame rate
    WriteData(0x00);
    WriteData(0x1B); //70

    //Gamma
    WriteCommand(ENABLE_3_GAMMA); // 3Gamma Function Disable
    WriteData(0x08);

    WriteCommand(GAMMA_SET);
    WriteData(0x01); // gamma set 4 gamma curve 01/02/04/08

    WriteCommand(POSITIVE_GAMMA_CORRECT); //positive gamma correction
    WriteData(0x1f);
    WriteData(0x1a);
    WriteData(0x18);
    WriteData(0x0a);
    WriteData(0x0f);
    WriteData(0x06);
    WriteData(0x45);
    WriteData(0x87);
    WriteData(0x32);
    WriteData(0x0a);
    WriteData(0x07);
    WriteData(0x02);
    WriteData(0x07);
    WriteData(0x05);
    WriteData(0x00);

    WriteCommand(NEGATIVE_GAMMA_CORRECT); //negative gamma correction
    WriteData(0x00);
    WriteData(0x25);
    WriteData(0x27);
    WriteData(0x05);
    WriteData(0x10);
    WriteData(0x09);
    WriteData(0x3a);
    WriteData(0x78);
    WriteData(0x4d);
    WriteData(0x05);
    WriteData(0x18);
    WriteData(0x0d);
    WriteData(0x38);
    WriteData(0x3a);
    WriteData(0x1f);

    //Set the column and page ranges
    WriteCommand(COLUMN_ADDRESS_SET); // column set
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0xEF);
    WriteCommand(PAGE_ADDRESS_SET); // page address set
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x01);
    WriteData(0x3F);

    //entry mode set
    WriteCommand(ENTRY_MODE_SET);
    WriteData(0x07);

    // display function control
    WriteCommand(DISPLAY_FUNCTION);
    WriteData(0x0a);
    WriteData(0x82);
    WriteData(0x27);
    WriteData(0x00);

    //MDT- for reading data back
    WriteCommand(INTERFACE_CONTROL);
    WriteData(0x01);
    WriteData(0x01);
    WriteData(0x00);

    // Sleep Out
    WriteCommand(SLEEP_OUT);
    DelayMs(120); //Wait to wake up

    //Display On
    WriteCommand(DISPLAY_ON); //Turn the display on

    SetColor(0);
    ClearDevice();
}

/*********************************************************************
 * Function: void PutPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: puts pixel
 *
 * Note: none
 *
 ********************************************************************/
void PutPixel(SHORT x, SHORT y) {
    if (_clipRgn) {
        if (x < _clipLeft)
            return;
        if (x > _clipRight)
            return;
        if (y < _clipTop)
            return;
        if (y > _clipBottom)
            return;
    }

    SetColumnRange(x, x);
    SetPageRange(y, y);

    WriteCommand(MEMORY_WRITE);
    WriteData(_color);
}

/*********************************************************************
 * Function: WORD GetPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: pixel color
 *
 * Side Effects: none
 *
 * Overview: returns pixel color at x,y position
 *
 * Note: none
 *
 ********************************************************************/
WORD GetPixel(SHORT x, SHORT y) {
    WORD dat;

    GetPixelArray(x, y, &dat, 1, 1);

    return dat;
}

#ifdef USE_TRANSPARENT_COLOR

/*********************************************************************
 * Function:  void TransparentColorEnable(GFX_COLOR color)
 *
 * Overview: Sets current transparent color.
 *
 * PreCondition: none
 *
 * Input: color - Color value chosen.
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TransparentColorEnable(GFX_COLOR color) {
    _colorTransparent = color;
    _colorTransparentEnable = TRANSPARENT_COLOR_ENABLE;

}
#endif

/*********************************************************************
 * Function: WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom)
 *
 * PreCondition: none
 *
 * Input: left,top - top left corner coordinates,
 *        right,bottom - bottom right corner coordinates
 *
 * Output: For NON-Blocking configuration:
 *         - Returns 0 when device is busy and the shape is not yet completely drawn.
 *         - Returns 1 when the shape is completely drawn.
 *         For Blocking configuration:
 *         - Always return 1.
 *
 * Side Effects: none
 *
 * Overview: draws rectangle filled with current color
 *
 * Note: none
 *
 ********************************************************************/
WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom) {

    if (right < left || bottom < top)
        return 1;

#ifndef USE_NONBLOCKING_CONFIG
    while (IsDeviceBusy() != 0);
    /* Ready */
#else
    if (IsDeviceBusy() != 0)
        return (0);
#endif
    if (_clipRgn) {
        if (left < _clipLeft)
            left = _clipLeft;
        if (right > _clipRight)
            right = _clipRight;
        if (top < _clipTop)
            top = _clipTop;
        if (bottom > _clipBottom)
            bottom = _clipBottom;
    }

    SetColumnRange(left, right);
    SetPageRange(top, bottom);

    WriteCommand(MEMORY_WRITE);

    LONG count = ((LONG)(right - left + 1)*(LONG)(bottom - top + 1));

    //for (i = 0; i < ((right - left + 1)*(bottom - top + 1)); ++i) {
    while(count--)
    {
        WriteData(_color);
    }

    return (1);
}

/*********************************************************************
 * Function: void ClearDevice(void)
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: clears screen with current color
 *
 * Note: none
 *
 ********************************************************************/
void ClearDevice(void) {
    while (!Bar(0, 0, GetMaxX(), GetMaxY()));
}

/*********************************************************************
 * Function: SetClipRgn(left, top, right, bottom)
 *
 * Overview: Sets clipping region.
 *
 * PreCondition: none
 *
 * Input: left - Defines the left clipping region border.
 *         top - Defines the top clipping region border.
 *         right - Defines the right clipping region border.
 *         bottom - Defines the bottom clipping region border.
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void SetClipRgn(SHORT left, SHORT top, SHORT right, SHORT bottom) {
    _clipLeft = left;
    _clipTop = top;
    _clipRight = right;
    _clipBottom = bottom;

}

/*********************************************************************
 * Function: SetClip(control)
 *
 * Overview: Enables/disables clipping.
 *
 * PreCondition: none
 *
 * Input: control - Enables or disables the clipping.
 *            - 0: Disable clipping
 *            - 1: Enable clipping
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void SetClip(BYTE control) {
    _clipRgn = control;
}

/*********************************************************************
 * Function: IsDeviceBusy()
 *
 * Overview: Returns non-zero if LCD controller is busy
 *           (previous drawing operation is not completed).
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: Busy status.
 *
 * Side Effects: none
 *
 ********************************************************************/
WORD IsDeviceBusy(void) {
    return 0;
}

/*********************************************************************
 * Function: void GetPixelArray()
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data read
 *              yStart: The y starting position of the data read
 *              *desArray: The array of data to be read
 *              len: The length of the data to read
 *              dir: The direction of the read (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Reads an array of data of length len from the frame buffer
 *           on the SSD1963.  This function is used for alphablending.
 *
 * Note: none
 *
 ********************************************************************/
void GetPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *desArray, WORD len, WORD dir) {
    WORD tmp;
    SetColumnRange(xStart, GetMaxX());
    SetPageRange(yStart, GetMaxY());

    WriteCommand(MEMORY_READ);

    DisplayEnable();
    DeviceRead();
    while (len--) {
        tmp = DeviceRead();
        *desArray = (tmp & 0xF800) | ((tmp & 0x00FC) << 3);
        tmp = DeviceRead();
        *desArray |= (tmp & 0xFC00) >> 11;
        if (len > 1)
            ++desArray;
    }
    DisplayDisable();
}

/*********************************************************************
 * Function: void PutPixelArray()
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data write
 *              yStart: The y starting position of the data write
 *              *desArray: The array of data to be written
 *              len: The length of the data to write
 *              dir: The direction of the write (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Writes an array of data of length len to the frame buffer
 *           on the SSD1963.  This function is used for alphablending
 *
 * Note: none
 *
 ********************************************************************/
void PutPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *srcArray, WORD len, WORD dir) {
    SetColumnRange(xStart, GetMaxX());
    SetPageRange(yStart, GetMaxY());

    WriteCommand(MEMORY_WRITE);

    DisplayEnable();
    while (len--) {
        DeviceWrite(*srcArray);
        ++srcArray;
    }
    DisplayDisable();
}



#endif // #if (GFX_USE_DISPLAY_CONTROLLER_ILI9341)
