/*****************************************************************************
 *
 * This document contains proprietary information and except with
 * written permission of Displaytech, Ltd., such information shall
 * not be published or disclosed to others or used for any purpose
 * other than for the operation and maintenance of the equipment
 * and software with which it was procured, and the document shall
 * not be copied in whole or in part.
 *
 * Copyright 2008-2015 Displaytech, Ltd. All Rights Reserved.
 *
 *****************************************************************************
 *  Module for Microchip Graphics Library
 *  Ilitek. ILI9163 driver
 *****************************************************************************
 * FileName:        ILI9163_SPI.c
 * Processor:       PIC32
 * Compiler:        MPLAB C32
 * Company:         Displaytech Ltd.
 *
 * Date         Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 06/22/12     Initial Revision
 ****************************************************************************/
#include "GraphicsConfig.h"
#include "HardwareProfile.h"
#include "TimeDelay.h"

#if defined(USE_TCON_ILI9163)

#if defined (GFX_USE_DISPLAY_CONTROLLER_SSD1926)
#include "Graphics/SSD1926.h"
#elif defined (GFX_USE_DISPLAY_CONTROLLER_S1D13517)
#include "Graphics/S1D13517.h"
#elif defined (GFX_USE_DISPLAY_CONTROLLER_MCHP_DA210) || defined (GFX_USE_DISPLAY_CONTROLLER_DMA)
// do not need any driver header file
#else
#warning "This Timing Controller Driver is written for SSD1926, S1D13517 and Microchip Graphics Module driver. If you are not using those drivers you may need to re-write this driver."
#endif

/*
    This module can use SPI or bitbang the SPI transfer using
    the SSD1926 GPIO pins.
    This assumes that the timing controller uses SPI to
    receive Timing Programming data.
    Select the usage using the hardware profile.
 */
// BB means BitBang
#define BB_CS                   0x01
#define BB_SCL                  0x02
#define BB_SDO                  0x04
//#define BB_DC                   0x08
#define BB_BL                   0x10

#if defined (GFX_USE_DISPLAY_CONTROLLER_DMA)

#define TCON_CSLow()            (TCON_CS_LAT = 0)
#define TCON_CSHigh()           (TCON_CS_LAT = 1)
#define TCON_CLKLow()           (TCON_SCL_LAT = 0)
#define TCON_CLKHigh()          (TCON_SCL_LAT = 1)
#define TCON_DataLow()          (TCON_SDO_LAT = 0)
#define TCON_DataHigh()         (TCON_SDO_LAT = 1)

#define TCON_SetCommand()       (TCON_DC_LAT = 0)
#define TCON_SetData()          (TCON_DC_LAT = 1)

// set the IOs used to outputs. and initialize them all to "1"
// set up the TRIS first for outputs, then set the pins
// to digital in case it is needed and initialize the signals
// to all high.
#define InitBitBangedIO() {                     \
                                TCON_CS_TRIS = 0;   \
                                TCON_SCL_TRIS = 0;  \
                                TCON_SDO_TRIS = 0;  \
                                TCON_DC_TRIS = 0;   \
                              }

#elif defined (GFX_USE_DISPLAY_CONTROLLER_MCHP_DA210)

#define TCON_CSLow()            (TCON_CS_LAT = 0)
#define TCON_CSHigh()           (TCON_CS_LAT = 1)
#define TCON_CLKLow()           (TCON_SCL_LAT = 0)
#define TCON_CLKHigh()          (TCON_SCL_LAT = 1)
#define TCON_DataLow()          (TCON_SDO_LAT = 0)
#define TCON_DataHigh()         (TCON_SDO_LAT = 1)

//    #define TCON_SetCommand()       (TCON_DC_LAT = 0)
//    #define TCON_SetData()          (TCON_DC_LAT = 1)

// set the IOs used to outputs. and initialize them all to "1"
// set up the TRIS first for outputs, then set the pins
// to digital in case it is needed and initialize the signals
// to all high.
#define InitBitBangedIO() {                     \
                                TCON_CS_TRIS = 0;   \
                                TCON_SCL_TRIS = 0;  \
                                TCON_SDO_TRIS = 0;  \
                                TCON_SDO_DIG();     \
                                TCON_CSHigh();      \
                                TCON_CLKHigh();     \
                                TCON_DataHigh();    \
                              }
//TCON_DC_TRIS = 0;
//TCON_CS_DIG();
//TCON_SCL_DIG();

//TCON_DC_DIG();

//                                TCON_SetData();
//                              }

// end of #if defined (USE_DISPLAY_CONTROLLER_MCHP_DA210)


#elif defined (GFX_USE_DISPLAY_CONTROLLER_SSD1926)

// use the bitbang version using SSD1926 GPIO pins

#define TCON_CSLow()            (GfxTconSetIO(BB_CS, 0))
#define TCON_CSHigh()           (GfxTconSetIO(BB_CS, 1))
#define TCON_CLKLow()           (GfxTconSetIO(BB_SCL, 0))
#define TCON_CLKHigh()          (GfxTconSetIO(BB_SCL, 1))
#define TCON_DataLow()          (GfxTconSetIO(BB_SDO, 0))
#define TCON_DataHigh()         (GfxTconSetIO(BB_SDO, 1))

//#define TCON_SetCommand()       (GfxTconSetIO(BB_DC, 0))
//#define TCON_SetData()          (GfxTconSetIO(BB_DC, 1))

// this is only needed here since the controller IO's are used
// instead of the IO's from the PIC device.
#define SetCtrlBitBangedIO(addr, data)    (SetReg(addr, data))

// set the GPIO of SSD1926 to as outputs. (used for ILI9163 TCON signals)
// and initialize them all to "1"
#define InitBitBangedIO() {                                     \
                                SetCtrlBitBangedIO(0xA8, 0x1F);     \
                                SetCtrlBitBangedIO(0xAC, 0x1F);     \
                              }
// end of #elif defined (USE_TCON_ILI9163)

#elif defined (GFX_USE_DISPLAY_CONTROLLER_S1D13517)

// use the bitbang version using SSD1926 GPIO pins

#define TCON_CSLow()            (GfxTconSetIO(CS, 0))
#define TCON_CSHigh()           (GfxTconSetIO(CS, 1))
#define TCON_CLKLow()           (GfxTconSetIO(SCL, 0))
#define TCON_CLKHigh()          (GfxTconSetIO(SCL, 1))
#define TCON_DataLow()          (GfxTconSetIO(SDO, 0))
#define TCON_DataHigh()         (GfxTconSetIO(SDO, 1))

#define TCON_SetCommand()       (GfxTconSetIO(DC, 0))
#define TCON_SetData()          (GfxTconSetIO(DC, 1))

// this is only needed here since the controller IO's are used
// instead of the IO's from the PIC device.
#define SetCtrlBitBangedIO(addr, data)    (SetReg(addr, data))

// set the GPIO of SSD1926 to as outputs. (used for ILI9163 TCON signals)
// and initialize them all to "1"
#define InitBitBangedIO() {                                     \
                                SCL_TRIS = 0;                       \
                                SDO_TRIS = 0;                       \
                              }

// end of #elif defined (USE_DISPLAY_CONTROLLER_S1D13517)

#endif

/************************************************************************
 * Function: void GfxTconSetIO(BYTE mask, BYTE level)
 *
 * Overview: This sets the IO specified by mask to the value set by
 *           level.
 *
 * Input: mask - specifies the IO to be toggles.
 *         level - specifies the logic where the IO will be set.
 *
 * Output: none
 *
 ************************************************************************/
void GfxTconSetIO(BYTE mask, BYTE level) {
#if defined (GFX_USE_DISPLAY_CONTROLLER_MCHP_DA210) || defined (GFX_USE_DISPLAY_CONTROLLER_DMA)

    switch (mask) {
        case BB_CS: (level == 1 ? TCON_CSHigh(): TCON_CSLow());
            break;

        case BB_SCL: (level == 1 ? TCON_CLKHigh(): TCON_CLKLow());
            break;

        case BB_SDO: (level == 1 ? TCON_DataHigh(): TCON_DataLow());
            break;

            //case BB_DC:    (level == 1 ? TCON_SetData() : TCON_SetCommand());
            //break;
        default:
            break;
    }

    Nop();

#elif defined (GFX_USE_DISPLAY_CONTROLLER_SSD1926)

    static BYTE value = 0xFF;

    if (level == 0) {
        value &= ~mask;
    } else {
        value |= mask;
    }

    SetCtrlBitBangedIO(0xAC, value);

#elif defined (GFX_USE_DISPLAY_CONTROLLER_S1D13517)

    switch (mask) {
        case CS:
            temp = GetReg(REG6E_GPO_1);
            if (level == 1) temp |= 0x02;
            else temp &= 0xFD;
            SetReg(REG6E_GPO_1, temp);
            break;

        case SCL: SCL_PORT = level;
            break;

        case SDO: SDO_PORT = level;
            break;

        case DC:
            temp = GetReg(REG6E_GPO_1);
            if (level == 1) temp |= 0x04;
            else temp &= 0xFB;
            SetReg(REG6E_GPO_1, temp);
            break;
    }

    Nop();

#else

#error "This TCON_ILI9163 is written for SSD1926, S1D13517 and Microchip Graphics Module driver. If you are not using those drivers you may need to re-write this driver and remove this error message."


#endif

}

/************************************************************************
 * Function: void GfxTconWriteByte(BYTE value)
 *
 * Overview: This writes a bytes to SPI.
 *
 * Input: value - the byte data to be written.
 *
 * Output: none
 *
 ************************************************************************/
void GfxTconWriteByte(BYTE value) {
    BYTE mask;

    mask = 0x80;
    while (mask) {
        GfxTconSetIO(BB_SCL, 0);

        if (mask & value) {
            GfxTconSetIO(BB_SDO, 1);
        } else {
            GfxTconSetIO(BB_SDO, 0);
        }
        Delay10us(1);
        GfxTconSetIO(BB_SCL, 1);
        Delay10us(1);
        mask >>= 1;
    }
}

/************************************************************************
 * Function: void GfxTconWriteCommand(BYTE value)
 *
 * Overview: This writes a word to SPI by calling the write byte
 *           routine.
 *
 * Input:
 *        value - The value that will be written to the register.
 *
 * Output: none
 *
 ************************************************************************/
void GfxTconWriteCommand(BYTE value) {
    //The CS is already high
    GfxTconSetIO(BB_CS, 0);

    //Send the D/CX bit
    GfxTconSetIO(BB_SCL, 0);
    GfxTconSetIO(BB_SDO, 0);
    GfxTconSetIO(BB_SCL, 1);
    Delay10us(1);

    //Send the command
    GfxTconWriteByte(value);

    GfxTconSetIO(BB_SCL, 0);
    Delay10us(1);
    GfxTconSetIO(BB_CS, 1);
}

//Read one byte of data back

BYTE GfxTconReadData(BYTE cmd) {
    BYTE mask;
    BYTE value;

    //Set the CS low
    GfxTconSetIO(BB_CS, 0);
    Delay10us(1);

    //Send the D/CX bit
    GfxTconSetIO(BB_SCL, 0);
    GfxTconSetIO(BB_SDO, 0);
    GfxTconSetIO(BB_SCL, 1);
    Delay10us(1);

    //Send the command
    GfxTconWriteByte(cmd); //Writes 8 bits

    //Set the clock low for the final bit
    GfxTconSetIO(BB_SCL, 0);

    //Tri-state the pin
    TCON_SDO_LAT = 0;
    TCON_SDO_TRIS = 1;
    Delay10us(1);

    mask = 0x80;
    value = 0;
    while (mask) {
        GfxTconSetIO(BB_SCL, 1);
        if (PORTBbits.RB1) //Port B1 is the SDA pin
        {
            value |= mask;
        }
        Delay10us(1);
        GfxTconSetIO(BB_SCL, 0);
        Delay10us(1);
        mask >>= 1;
    }

    //Set the CS high
    GfxTconSetIO(BB_CS, 1);

    //Reset the tri-state
    TCON_SDO_TRIS = 0;
    return value;
}

/************************************************************************
 * Function: void GfxTconWriteParameter(BYTE value)
 *
 * Overview: This writes a word to SPI by calling the write byte
 *           routine.
 *
 * Input:
 *        value - The value that will be written to the register.
 *
 * Output: none
 *
 ************************************************************************/
void GfxTconWriteParameter(BYTE value) {
    //The CS is already high
    GfxTconSetIO(BB_CS, 0);

    //Send the D/CX bit
    GfxTconSetIO(BB_SCL, 0);
    GfxTconSetIO(BB_SDO, 1);
    GfxTconSetIO(BB_SCL, 1);
    Delay10us(1);

    //Send the command
    GfxTconWriteByte(value);

    GfxTconSetIO(BB_SCL, 0);
    Delay10us(1);
    GfxTconSetIO(BB_CS, 1);
}

/************************************************************************
 * Function: void GfxTconInit(void)
 *
 * Overview: Initialize the IOs to implement Bitbanged SPI interface
 *           to connect to the Timing Controller through SPI.
 *
 * Input: none
 *
 * Output: none
 *
 ************************************************************************/
void GfxTconInit(void) {
    InitBitBangedIO();

    DisplayBacklightOn();
    DisplayBacklightConfig();

    //Hardware reset
    DisplayPowerOff();
    DelayMs(200);
    DisplayPowerOn();
    DelayMs(200);

    //Software Reset
    GfxTconWriteCommand(0x01);
    DelayMs(200);

    //RGB, set VIPF and
    GfxTconWriteCommand(0x0C);
    GfxTconWriteParameter(0x66);

    //RGB/BGR swap
    GfxTconWriteCommand(0x36);
    GfxTconWriteParameter(0x08);

    // Sleep Out
    GfxTconWriteCommand(0x11);
    DelayMs(200);

    //Display On
    GfxTconWriteCommand(0x29);

    TCON_CLKLow();

}



#endif // #if defined (USE_TCON_ILI9163)


