/*****************************************************************************
 *
 * This document contains proprietary information and except with
 * written permission of Displaytech, Ltd., such information shall
 * not be published or disclosed to others or used for any purpose
 * other than for the operation and maintenance of the equipment
 * and software with which it was procured, and the document shall
 * not be copied in whole or in part.
 *
 * Copyright 2008-2015 Displaytech, Ltd. All Rights Reserved.
 *
 *****************************************************************************
 *  Module for Microchip Graphics Library
 *  Ilitek ILI9163 LCD controller driver
 *****************************************************************************
 * FileName:        ILI9163.c
 * Processor:       PIC32
 * Compiler:        MPLAB C32
 * Company:         Displaytech Ltd.
 *
 * Date         Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 06/22/12     Initial Revision
 ****************************************************************************/
#include "HardwareProfile.h"

#if defined(GFX_USE_DISPLAY_CONTROLLER_ILI9163)
#ifndef USE_16BIT_PMP
#error("The ILI9163 is only compatible with 16 bit PMP");
#endif
#ifndef __PIC32MX__
#error("The ILI9163 is only compatible with the PIC32");
#endif

#include "Compiler.h"
#include "TimeDelay.h"
#include "Graphics/DisplayDriver.h"
#include "ILI9163.h"
#include "Graphics/gfxtcon.h"
#include "Graphics/Primitive.h"


#if defined (USE_GFX_PMP)
#include "Graphics/gfxpmp.h"
#elif defined (USE_GFX_EPMP)
#include "Graphics/gfxepmp.h"
#endif

// Color
GFX_COLOR _color;
#ifdef USE_TRANSPARENT_COLOR
GFX_COLOR _colorTransparent;
SHORT _colorTransparentEnable;
#endif

// Clipping region control
SHORT _clipRgn;

// Clipping region borders
SHORT _clipLeft;
SHORT _clipTop;
SHORT _clipRight;
SHORT _clipBottom;

#define RED8(color16)   (BYTE) ((color16 & 0xF800) >> 8)
#define GREEN8(color16) (BYTE) ((color16 & 0x07E0) >> 3)
#define BLUE8(color16)  (BYTE) ((color16 & 0x001F) << 3)

/////////////////////// LOCAL FUNCTIONS PROTOTYPES ////////////////////////////
void WriteCommand(BYTE command);
void WriteData(WORD data);
void SetPageRange(WORD start, WORD end);
void SetColumnRange(WORD start, WORD end);

#ifdef USE_PALETTE
extern void *_palette;
static BYTE PaletteBpp = 16;
extern BYTE blPaletteChangeError;
extern void *pPendingPalette;
extern WORD PendingStartEntry;
extern WORD PendingLength;
#endif

//Set the column memory range

void SetColumnRange(WORD start, WORD end) {
    WriteCommand(COLUMN_ADDRESS_SET); //Set Column Address
    WriteData((BYTE) (start >> 8)); //Start address [15:8]
    WriteData((BYTE) start); //Start address [7:0]
    WriteData((BYTE) (end >> 8)); //End address [15:8]
    WriteData((BYTE) end); //End address [7:0]
}

//Set the page memory range

void SetPageRange(WORD start, WORD end) {
    WriteCommand(PAGE_ADDRESS_SET); //Set Column Address
    WriteData((BYTE) (start >> 8)); //Start address [15:8]
    WriteData((BYTE) start); //Start address [7:0]
    WriteData((BYTE) (end >> 8)); //End address [15:8]
    WriteData((BYTE) end); //End address [7:0]
}

//Write a command

void WriteCommand(BYTE command) {
    //Set the CS Low
    DisplayEnable();

    //Set the RS pin LOW
    DisplaySetCommand();

    //Dump the command byte into the PMP
    PMDIN1 = (WORD) command;

    //Wait for the transmission end
    PMPWaitBusy();

    //Set the RS high, next bytes will likely be data
    DisplaySetData();

    //Set the CS high
    DisplayDisable();
}

//Write 1 word of data (after command)

void WriteData(WORD data) {
    //Set the CS low
    DisplayEnable();

    //Set the RS pin high
    DisplaySetData();

    //Dump the data into the PMP
    PMDIN1 = data;

    //Wait for the transmission end
    PMPWaitBusy();

    //Set the CS high
    DisplayDisable();
}

WORD ReadData(void)
{
    WORD data;

    //Set the CS low
    DisplayEnable();

    //Dump the data into the PMP
    data = DeviceRead();

    //Set the CS high
    DisplayDisable();
}

/*********************************************************************
 * Function:  void ResetDevice()
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: resets LCD, initializes PMP
 *
 * Note: none
 *
 ********************************************************************/
void ResetDevice(void) {
    /////////////////////////////////////////////////////////////////////
    // Initialize the device
    /////////////////////////////////////////////////////////////////////
    DriverInterfaceInit();

    DisplayResetEnable();
    DisplayResetConfig();
    DelayMs(100);
    DisplayResetDisable();
    DelayMs(100);

    DisplayBacklightOn();
    DisplayBacklightConfig();

    // give time for the controller to power up
    DelayMs(250);

    //Software Reset
    WriteCommand(SOFTWARE_RESET);

    //Supposed to wait at least 120ms, wait 250 to be safe
    DelayMs(250);

    //Pixel format
    WriteCommand(PIXEL_FORMAT_SET);
    WriteData(0x05); //16-bit 5-6-5

    //Memory access
    WriteCommand(MEMORY_ACCESS_CTRL);
    #if (DISP_HOR_RESOLUTION == 160) //160x128
    WriteData(0x0028); //Swap row/columns, RGB/BGR swap
    #else //128x160
        WriteData(0x00C8); //Swap row/columns, RGB/BGR swap
    #endif

    WriteCommand(0x26);    //Set Default Gamma
    WriteData(0x04);

    WriteCommand(0xf2);     //E0h & E1h Enable/Disable
    WriteData(0x01);
    WriteCommand(0xE0);
    WriteData(0x3F);
    WriteData(0x25);
    WriteData(0x1C);
    WriteData(0x1E);
    WriteData(0x20);
    WriteData(0x12);
    WriteData(0x2A);
    WriteData(0x90);
    WriteData(0x24);
    WriteData(0x11);
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x00);

    WriteCommand(0xE1);    //Negative Gamma Setting
    WriteData(0x20);
    WriteData(0x20);
    WriteData(0x20);
    WriteData(0x20);
    WriteData(0x05);
    WriteData(0x00);
    WriteData(0x15);
    WriteData(0xA7);
    WriteData(0x3D);
    WriteData(0x18);
    WriteData(0x25);
    WriteData(0x2A);
    WriteData(0x2B);
    WriteData(0x2B);
    WriteData(0x3A);

    WriteCommand(0xB0);
    WriteData(0x00);//08
    //WriteData(0x08);//14
    WriteCommand(0xB1);
    WriteData(0x08);//08
    WriteData(0x08);//14
    WriteCommand(0xc0);     //Set VRH1[4:0] & VC[2:0] for VCI1 & GVDD
    WriteData(0x0a); //0A //14
    WriteData(0x02);
    WriteCommand(0xC1);    //Set BT[2:0] for AVDD & VCL & VGH & VGL
    WriteData(0x02);
    WriteCommand(0xC5);     //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
    WriteData(0x1A);   //1a
    WriteData(0x4A);  //4a
    WriteCommand(0xC6);     //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
    WriteData(0x13);   //1a
    WriteData(0x06);  //4a
    WriteCommand(0xC7);     //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
    WriteData(0xFF);   //40   modify the fliker

    // Sleep Out
    WriteCommand(SLEEP_OUT);
    DelayMs(120); //Wait to wake up

    //Display On
    WriteCommand(DISPLAY_ON); //Turn the display on

    /////////////////////////////////////////////////////////////////////
    // PALETTE INIT
    /////////////////////////////////////////////////////////////////////
#ifdef USE_PALETTE
    PaletteInit();
#endif

    /////////////////////////////////////////////////////////////////////
    // Clear the display buffer with all zeros so the display will not
    // show garbage data when initially powered up
    /////////////////////////////////////////////////////////////////////
    SetColor(0);
    ClearDevice();
}

/*********************************************************************
 * Function: void PutPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: puts pixel
 *
 * Note: none
 *
 ********************************************************************/
void PutPixel(SHORT x, SHORT y) {
    if (_clipRgn) {
        if (x < _clipLeft)
            return;
        if (x > _clipRight)
            return;
        if (y < _clipTop)
            return;
        if (y > _clipBottom)
            return;
    }

    SetColumnRange(x, x);
    SetPageRange(y, y);

    WriteCommand(MEMORY_WRITE);
    WriteData(_color);
}

/*********************************************************************
 * Function: WORD GetPixel(SHORT x, SHORT y)
 *
 * PreCondition: none
 *
 * Input: x,y - pixel coordinates
 *
 * Output: pixel color
 *
 * Side Effects: none
 *
 * Overview: returns pixel color at x,y position
 *
 * Note: none
 *
 ********************************************************************/
WORD GetPixel(SHORT x, SHORT y) {
    WORD value;

    SetColumnRange(x, x);
    SetPageRange(y, y);

    WriteCommand(MEMORY_READ);

    value = ReadData();

    return (value);
}

/*********************************************************************
 * Function: void GetPixelArray()
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data read
 *              yStart: The y starting position of the data read
 *              *desArray: The array of data to be read
 *              len: The length of the data to read
 *              dir: The direction of the read (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Reads an array of data of length len from the frame buffer
 *           on the SSD1963.  This function is used for alphablending.
 *
 * Note: none
 *
 ********************************************************************/
void GetPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *desArray, WORD len, WORD dir)
{
    SetColumnRange(xStart, GetMaxX());
    SetPageRange(yStart, GetMaxY());

    WriteCommand(MEMORY_READ);

    DisplayEnable();
    while (len--)
    {
        *desArray = DeviceRead();
        ++desArray;
    }
    DisplayDisable();
}

/*********************************************************************
 * Function: void PutPixelArray()
 *
 * PreCondition: none
 *
 * Input:       xStart: The x starting position of the data write
 *              yStart: The y starting position of the data write
 *              *desArray: The array of data to be written
 *              len: The length of the data to write
 *              dir: The direction of the write (unused)
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: Writes an array of data of length len to the frame buffer
 *           on the SSD1963.  This function is used for alphablending
 *
 * Note: none
 *
 ********************************************************************/
void PutPixelArray(SHORT xStart, SHORT yStart, GFX_COLOR *srcArray, WORD len, WORD dir)
{
    SetColumnRange(xStart, GetMaxX());
    SetPageRange(yStart, GetMaxY());

    WriteCommand(MEMORY_WRITE);

    DisplayEnable();
    while (len--)
    {
        DeviceWrite(*srcArray);
        ++srcArray;
    }
    DisplayDisable();
}

#ifdef USE_TRANSPARENT_COLOR

/*********************************************************************
 * Function:  void TransparentColorEnable(GFX_COLOR color)
 *
 * Overview: Sets current transparent color.
 *
 * PreCondition: none
 *
 * Input: color - Color value chosen.
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void TransparentColorEnable(GFX_COLOR color) {
    _colorTransparent = color;
    _colorTransparentEnable = TRANSPARENT_COLOR_ENABLE;

}
#endif

#ifndef USE_PALETTE
// "In SSD1926 2D-Acceleration is not supported in Palette mode. Use Line function of Primitive layer"

#endif // #ifndef USE_PALETTE

/*********************************************************************
 * Function: WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom)
 *
 * PreCondition: none
 *
 * Input: left,top - top left corner coordinates,
 *        right,bottom - bottom right corner coordinates
 *
 * Output: For NON-Blocking configuration:
 *         - Returns 0 when device is busy and the shape is not yet completely drawn.
 *         - Returns 1 when the shape is completely drawn.
 *         For Blocking configuration:
 *         - Always return 1.
 *
 * Side Effects: none
 *
 * Overview: draws rectangle filled with current color
 *
 * Note: none
 *
 ********************************************************************/
WORD Bar(SHORT left, SHORT top, SHORT right, SHORT bottom) {
    DWORD i;

    if(right < left || bottom < top)
        return 1;

#ifndef USE_NONBLOCKING_CONFIG
    while (IsDeviceBusy() != 0);
    /* Ready */
#else
    if (IsDeviceBusy() != 0)
        return (0);
#endif
    if (_clipRgn) {
        if (left < _clipLeft)
            left = _clipLeft;
        if (right > _clipRight)
            right = _clipRight;
        if (top < _clipTop)
            top = _clipTop;
        if (bottom > _clipBottom)
            bottom = _clipBottom;
    }

    SetColumnRange(left, right);
    SetPageRange(top, bottom);

    WriteCommand(MEMORY_WRITE);

    for (i = 0; i < ((right - left + 1)*(bottom - top + 1)); ++i) {
        WriteData(_color);
    }

    return (1);
}

/*********************************************************************
 * Function: void ClearDevice(void)
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 * Overview: clears screen with current color
 *
 * Note: none
 *
 ********************************************************************/
void ClearDevice(void) {
    while (!Bar(0, 0, GetMaxX(), GetMaxY()));
}

/*********************************************************************
 * Function: SetClipRgn(left, top, right, bottom)
 *
 * Overview: Sets clipping region.
 *
 * PreCondition: none
 *
 * Input: left - Defines the left clipping region border.
 *         top - Defines the top clipping region border.
 *         right - Defines the right clipping region border.
 *         bottom - Defines the bottom clipping region border.
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void SetClipRgn(SHORT left, SHORT top, SHORT right, SHORT bottom) {
    _clipLeft = left;
    _clipTop = top;
    _clipRight = right;
    _clipBottom = bottom;

}

/*********************************************************************
 * Function: SetClip(control)
 *
 * Overview: Enables/disables clipping.
 *
 * PreCondition: none
 *
 * Input: control - Enables or disables the clipping.
 *            - 0: Disable clipping
 *            - 1: Enable clipping
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void SetClip(BYTE control) {
    _clipRgn = control;
}

/*********************************************************************
 * Function: IsDeviceBusy()
 *
 * Overview: Returns non-zero if LCD controller is busy
 *           (previous drawing operation is not completed).
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: Busy status.
 *
 * Side Effects: none
 *
 ********************************************************************/
WORD IsDeviceBusy(void) {
    //return ((GetReg(REG_2D_220) & 0x01) == 0);
    return 0;
}

#ifdef USE_PALETTE

/*********************************************************************
 * Function: void PaletteInit(void)
 *
 * Overview: Initializes the CLUT.
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: Drawing mode will change to support palettes
 *
 ********************************************************************/
void PaletteInit(void) {
    PaletteBpp = 16;
    blPaletteChangeError = 0;
    pPendingPalette = NULL;
    PendingStartEntry = 0;
    PendingLength = 0;
}

/*********************************************************************
 * Function: BYTE SetPaletteBpp(BYTE bpp)
 *
 * Overview: Sets the CLUT's number of valid entries.
 *
 * PreCondition: PaletteInit() must be called before.
 *
 * Input: bpp -> Bits per pixel
 *
 * Output: Status: Zero -> Success, Non-zero -> Error.
 *
 * Side Effects: Drawing mode will change to support palettes
 *
 ********************************************************************/
BYTE SetPaletteBpp(BYTE bpp) {
    switch (bpp) {
            /*case 1: SetReg(REG_DISPLAY_MODE,0x00);
                    break;

            case 2: SetReg(REG_DISPLAY_MODE,0x01);
                    break;

            case 4: SetReg(REG_DISPLAY_MODE,0x02);
                    break;*/
        case 8: SetReg(REG_DISPLAY_MODE, 0x03);
            break;
        case 16: SetReg(REG_DISPLAY_MODE, 0x04);
            break;
        default: SetReg(REG_DISPLAY_MODE, 0x04);
            PaletteBpp = 16;
            return (-1);
    }

    PaletteBpp = bpp;

    /////////////////////////////////////////////////////////////////////
    // ROTATION MODE
    /////////////////////////////////////////////////////////////////////
#if (DISP_ORIENTATION == 0)
#define WIN_START_ADDR  0ul
#define ROTATION        0

#elif (DISP_ORIENTATION == 90)
#ifndef USE_PALETTE
#define WIN_START_ADDR  ((((DWORD) GetMaxX() + 1) >> 1) - 1)
#else
#define WIN_START_ADDR  (((((DWORD) GetMaxX() + 1) * PaletteBpp) >> 5) - 1)
#endif
#define ROTATION    1

#elif (DISP_ORIENTATION == 180)
#ifndef USE_PALETTE
#define WIN_START_ADDR  (((((DWORD) GetMaxX() + 1) * (GetMaxY() + 1)) >> 1) - 1)
#else
#define WIN_START_ADDR  (((((DWORD) GetMaxX() + 1) * (GetMaxY() + 1) * PaletteBpp) >> 5) - 1)
#endif
#define ROTATION    2

#elif (DISP_ORIENTATION == 270)
#ifndef USE_PALETTE
#define WIN_START_ADDR  ((((DWORD) GetMaxX() + 1) * GetMaxY()) >> 1)
#else
#define WIN_START_ADDR  ((((DWORD) GetMaxX() + 1) * GetMaxY() * PaletteBpp) >> 5)
#endif
#define ROTATION    3
#endif

    /////////////////////////////////////////////////////////////////////
    // Special Effects Register (reg 71h)
    /////////////////////////////////////////////////////////////////////
#ifndef USE_PALETTE
    SetReg(REG_SPECIAL_EFFECTS, 0x40 | ROTATION);
#else
    SetReg(REG_SPECIAL_EFFECTS, 0x00 | ROTATION);
#endif

    /////////////////////////////////////////////////////////////////////
    // Main Window Display Start Address (regs 74h, 75h, 76h)
    /////////////////////////////////////////////////////////////////////
    SetReg(REG_MAIN_WIN_DISP_START_ADDR0, WIN_START_ADDR & 0x00FF);
    SetReg(REG_MAIN_WIN_DISP_START_ADDR1, (WIN_START_ADDR >> 8) & 0x00FF);
    SetReg(REG_MAIN_WIN_DISP_START_ADDR2, (WIN_START_ADDR >> 16) & 0x00FF);

    /////////////////////////////////////////////////////////////////////
    // Main Window Display Offset (regs 78h, 79h)
    /////////////////////////////////////////////////////////////////////
#ifndef USE_PALETTE
#define WIN_OFFSET  ((GetMaxX() + 1) >> 1)
#else
#define WIN_OFFSET  (((GetMaxX() + 1) * PaletteBpp) >> 5)
#endif
    SetReg(REG_MAIN_WIN_ADDR_OFFSET0, WIN_OFFSET & 0x00FF);
    SetReg(REG_MAIN_WIN_ADDR_OFFSET1, (WIN_OFFSET >> 8) & 0x00FF);

    return (0);
}

/*********************************************************************
 * Function: void EnablePalette(void)
 *
 * Overview: Enables the Palette mode
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects:
 *
 ********************************************************************/
void EnablePalette(void) {
    SetPaletteBpp(PaletteBpp);
}

/*********************************************************************
 * Function: void DisablePalette(void)
 *
 * Overview: Disables the Palette mode
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects:
 *
 ********************************************************************/
void DisablePalette(void) {
    SetPaletteBpp(16);
}

/*********************************************************************
 * Function: BYTE IsPaletteEnabled(void)
 *
 * Overview: Returns if the Palette mode is enabled or not
 *
 * PreCondition: none
 *
 * Input: none
 *
 * Output: Enabled -> 1, Disabled -> 0
 *
 * Side Effects:
 *
 ********************************************************************/
BYTE IsPaletteEnabled(void) {
    return ((PaletteBpp == 16) ? 0 : 1);
}

/*********************************************************************
 * Function: void StartVBlankInterrupt(void)
 *
 * Overview: Sets up the Vertical Blanking Interrupt
 *
 * PreCondition: Interrupts must be enabled
 *
 * Input: none
 *
 * Output: none
 *
 * Side Effects: none
 *
 ********************************************************************/
void StartVBlankInterrupt(void) {

    /* We don't use Vertical Blanking Interrupt in SSD1926 */
    if (pPendingPalette != NULL) {
        blPaletteChangeError = SetPalette(pPendingPalette, PendingStartEntry, PendingLength);
        if (!blPaletteChangeError) {
            _palette = pPendingPalette;
        }
    }
}

/*********************************************************************
 * Function: BYTE SetPaletteFlash(PALETTE_ENTRY *pPaletteEntry, WORD startEntry, WORD length)
 *
 * Overview: Loads the palettes from the flash immediately.
 *
 * PreCondition: PaletteInit() must be called before.
 *
 * Input: pPaletteEntry   - Pointer to the palette table in ROM
 *        startEntry      - Start entry to load (inclusive)
 *        length          - Number of entries
 *
 * Output: Status: Zero -> Success, Non-zero -> Error.
 *
 * Side Effects: There may be a slight flicker when the Palette entries
 *               are getting loaded one by one.
 *
 ********************************************************************/
BYTE SetPaletteFlash(PALETTE_ENTRY *pPaletteEntry, WORD startEntry, WORD length) {
    WORD counter;

    if ((pPaletteEntry == NULL) || ((startEntry + length) > 256)) {
        return (-1);
    }

    for (counter = 0; counter < length; counter++) {
        SetReg(REG_LUT_RED_WRITE_DATA, RED8(pPaletteEntry[counter].value));
        SetReg(REG_LUT_GREEN_WRITE_DATA, GREEN8(pPaletteEntry[counter].value));
        SetReg(REG_LUT_BLUE_WRITE_DATA, BLUE8(pPaletteEntry[counter].value));
        SetReg(REG_LUT_WRITE_ADDR, startEntry + counter);
    }

    return (0);
}

#endif

#endif // #if (GFX_USE_DISPLAY_CONTROLLER_SSD9163)
