/*****************************************************************************
 *
 * This document contains proprietary information and except with
 * written permission of Displaytech, Ltd., such information shall
 * not be published or disclosed to others or used for any purpose
 * other than for the operation and maintenance of the equipment
 * and software with which it was procured, and the document shall
 * not be copied in whole or in part.
 *
 * Copyright 2008-2014 Displaytech, Ltd. All Rights Reserved.
 *
 *****************************************************************************/

#include "GenericTypeDefs.h"
#include "HardwareProfile.h"
#include "Graphics/Graphics.h"
#include <string.h>
#include "FT5316.h"
#include "TimeDelay.h"

touch_packet_t touches[MAX_TOUCHES];
UINT8 numTouches = 0;
UINT8 gesture = 0;

#define i2cDelay() (Delay10us(1))

static BOOL startTransfer( void )
{
    I2C_SDA_HIGH();
    I2C_SCL_HIGH();
    i2cDelay();
    I2C_SDA_LOW();
    i2cDelay();
    I2C_SCL_LOW();

    return TRUE;
}

static BYTE i2cWrite( UINT8 data )
{
    BYTE mask = 0x80;
    BYTE ack = 0;

    I2C_SCL_LOW();
    while(mask)
    {
        i2cDelay();
        if(mask & data)
            I2C_SDA_HIGH();
        else
            I2C_SDA_LOW();

        I2C_SCL_HIGH();
        i2cDelay();
        I2C_SCL_LOW();
        mask >>= 1;
    }

    i2cDelay();
    //check for an ACK
    I2C_SDA_INPUT();
    I2C_SCL_HIGH();
    i2cDelay();
    ack = I2C_SDA_PORT;
    I2C_SCL_LOW();
    i2cDelay();
    I2C_SDA_OUTPUT();

    if(!ack)
        return TRUE;
    return FALSE;
}

static BYTE i2cRead(BYTE* data, BYTE  len)
{
    BYTE mask;
    BYTE value;

    while(len--)
    {
        //Start reading
        I2C_SDA_LOW();
        I2C_SDA_INPUT();
        I2C_SCL_LOW();

        mask = 0x80;
        value = 0x00;
        while(mask)
        {
            i2cDelay();
            I2C_SCL_HIGH();
            i2cDelay();
            if(I2C_SDA_PORT)
                value |= mask;
            I2C_SCL_LOW();
            mask >>= 1;
        }

         i2cDelay();
        //Send the ack (or nack)
        if(len)
            I2C_SDA_LOW();
        else
            I2C_SDA_HIGH();
        I2C_SDA_OUTPUT();
        I2C_SCL_HIGH();
        i2cDelay();
        I2C_SCL_LOW();

        *data = value;
        ++data;
    }
    return TRUE;
}

static void stopTransfer( void )
{
    I2C_SDA_LOW();
    I2C_SCL_HIGH();
    i2cDelay();
    I2C_SDA_HIGH();
    i2cDelay();
}

static void readRegister(BYTE reg, BYTE *dest, BYTE len)
{
    startTransfer();
    i2cWrite(I2C_WRITE_BYTE);
    i2cWrite(reg);

    startTransfer();
    i2cWrite(I2C_READ_BYTE);
    i2cRead(dest,len);
    stopTransfer();
}

static void byteReverse(UINT8* dest, UINT8* src, UINT8 len)
{
    src += (len - 1);

    while(len--)
        *dest++ = *src--;
}

void TouchHardwareInit(void)
{
    I2C_PIN_CONFIGURE();

    //Reset the device
    I2C_RESET_LOW();
    DelayMs(10);
    I2C_RESET_HIGH();
    DelayMs(200);

    memset(touches,0,sizeof(touches));
}

void TouchGetPosition(void)
{
    UINT8 i;
    UINT8 buf[4];

    readRegister(0x01,&gesture,1);

    //Get the number of touches
    readRegister(0x02,&numTouches,1);

    if(!numTouches || numTouches > MAX_TOUCHES)
    {
        for(i=0; i<MAX_TOUCHES; ++i)
        {
            touches[i].x = -1;
            touches[i].y = -1;
        }
        numTouches = 0;
        return;
    }

    for(i=0; i<numTouches; ++i)
    {
        startTransfer();
        i2cWrite(I2C_WRITE_BYTE);
        i2cWrite((UINT8)(3+(6*i)));
        startTransfer();
        i2cWrite(I2C_READ_BYTE);
        i2cRead(buf,4);
        stopTransfer();

        byteReverse((UINT8*)&touches[i].u32, buf, 4);
    }

    return;
}

SHORT TouchGetX(void)
{
    return (touches[0].x == 0x0FFF) ? -1 : touches[0].x;
}

SHORT TouchGetY(void)
{
    return (touches[0].y == 0x0FFF) ? -1 : touches[0].y;
}

SHORT TouchGetRawX(void)
{
    return (touches[0].x == 0x0FFF) ? -1 : touches[0].x;
}

SHORT TouchGetRawY(void)
{
    return (touches[0].y == 0x0FFF) ? -1 : touches[0].y;
}

SHORT mchpTouchScreenVersion = 0x0000;
void TouchStoreCalibration(void) { return; }
void TouchLoadCalibration(void)  { return; }
void TouchCalHWGetPoints(void) { return; }
