/*****************************************************************************
 *
 * This document contains proprietary information and except with
 * written permission of Displaytech, Ltd., such information shall
 * not be published or disclosed to others or used for any purpose
 * other than for the operation and maintenance of the equipment
 * and software with which it was procured, and the document shall
 * not be copied in whole or in part.
 *
 * Copyright 2008-2016 Displaytech, Ltd. All Rights Reserved.
 *
 * Company:         Displaytech Ltd.
 *
 * Date         Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * 02/5/15     Initial Revision
 ****************************************************************************/

#include <plib.h>

// Configuration Bits
#pragma config FNOSC    = PRIPLL        // Oscillator Selection
#pragma config FPLLIDIV = DIV_2         // PLL Input Divider (PIC32 Starter Kit: use divide by 2 only)
#pragma config FPLLMUL  = MUL_20        // PLL Multiplier
#pragma config FPLLODIV = DIV_1         // PLL Output Divider
#pragma config FPBDIV   = DIV_1         // Peripheral Clock divisor
#pragma config FWDTEN   = OFF           // Watchdog Timer
#pragma config WDTPS    = PS1           // Watchdog Timer Postscale
#pragma config FCKSM    = CSDCMD        // Clock Switching & Fail Safe Clock Monitor
#pragma config OSCIOFNC = OFF           // CLKO Enable
#pragma config POSCMOD  = XT            // Primary Oscillator
#pragma config IESO     = OFF           // Internal/External Switch-over
#pragma config FSOSCEN  = OFF           // Secondary Oscillator Enable
#pragma config CP       = OFF           // Code Protect
#pragma config BWP      = OFF           // Boot Flash Write Protect
#pragma config PWP      = OFF           // Program Flash Write Protect
#pragma config ICESEL   = ICS_PGx2      // ICE/ICD Comm Channel Select
#pragma config DEBUG    = OFF           // Debugger Disabled

//  The following is used by the main application
#define SYS_FREQ		(80000000)

//#define STANDARD_18 // Displaytech standard DT018TFT 1.8" display, uses 3 wire spi, Otherwise 4 wire with DT010TFT.  Both parts use the same ILI9163 Display Driver IC.  However Display resolutions are different.

#ifdef STANDARD_18
    #define THREE_WIRE_SPI /* Demo 3 wire SPI */
    #define  Lcm_H	 160
    #define  Lcm_W	 128
#else /* DT010TFT, uses 4 wire SPI */
    #define  Lcm_H	80
    #define  Lcm_W	160
#endif

// ------------------------------------------
//      I/O Macros
// ------------------------------------------
#define SclHigh()   LATEbits.LATE1=1;
#define SclLow()    LATEbits.LATE1=0;
#define DataHigh()  LATEbits.LATE0=1;
#define DataLow()   LATEbits.LATE0=0;
#define CsHigh()    LATDbits.LATD8=1;
#define CsLow()     LATDbits.LATD8=0;
#define ResetHigh() LATDbits.LATD5=1;
#define ResetLow()  LATDbits.LATD5=0;

#ifdef THREE_WIRE_SPI
#define FourWireDataLow()  
#define FourWireDataHigh()
#define FourWireCmdLow()  
#define FourWireCmdHigh() 
#else /* 4 Wire SPI */
#define FourWireDataLow()  LATAbits.LATA1=0; // CS/X (SCL)
#define FourWireDataHigh() LATAbits.LATA1=1; // CS/X (SCL)
#define FourWireCmdLow()  LATAbits.LATA1=0;
#define FourWireCmdHigh()  LATAbits.LATA1=1;
#endif /* THREE_WIRE_SPI */


void SpiWrite8Bits(unsigned char d);
void WriteCom(unsigned char Com);
void WriteData(unsigned char Data);
void WriteData1(unsigned int Data);
void BlockWrite(unsigned int Xstart, unsigned int Xend, unsigned int Ystart, unsigned int Yend);
void LCD_InitMain();
void Delay_ms(unsigned int count);
void ResetLCM();
void DispColor(unsigned int color);
void DispBand(void);


int main(void)
{
    SYSTEMConfig(SYS_FREQ, SYS_CFG_WAIT_STATES | SYS_CFG_PCACHE);

#ifndef THREE_WIRE_SPI
    // We are using RA1 which is used for JTAG.  Turn off JTAG so we
    // can use this line for D/CX in 4-Wire SPI.
    DDPCONbits.JTAGEN = 0;
    PORTSetPinsDigitalOut(IOPORT_A, BIT_1);
#endif

    PORTSetPinsDigitalOut(IOPORT_D, BIT_8 | BIT_4 | BIT_5);
    PORTSetPinsDigitalOut(IOPORT_E, BIT_0 | BIT_1);
    
    ResetLCM();
    LCD_InitMain();

    while (1)
    {
        DispColor(0xF800);
        Delay_ms(500);
        DispColor(0x07e0);
        Delay_ms(500);
        DispColor(0x001f);
        Delay_ms(500);
        DispBand();
        Delay_ms(500);
    }

}

void SpiWrite8Bits(unsigned char d)
{
    unsigned char j;

    for (j = 0; j < 8; j++)
    {
        if (d & 0x80)
        {
            DataHigh();
        }
        else
        {
            DataLow();
        }
        SclHigh();
        SclLow();
        d <<= 1;
    }
}


// Write Command
void WriteCom(unsigned char Com)
{
    CsLow();
#ifdef THREE_WIRE_SPI
    DataLow();
    SclHigh();
#else /* FOUR_WIRE_SPI */
    FourWireCmdLow(); // CS/X (SCL)
#endif
    
    SclLow();
    SpiWrite8Bits(Com);
    CsHigh();
}

// Write Data (8 bits)
void WriteData(unsigned char Data)
{
    CsLow();
#ifdef THREE_WIRE_SPI
    DataHigh();
    SclHigh();
#else /* FOUR_WIRE_SPI */
    FourWireDataHigh(); // CS/X (SCL)
#endif

    SclLow();
    SpiWrite8Bits(Data);
    CsHigh();
}

// Write Data (16 bits)
void WriteData1(unsigned int Data)
{
    WriteData(Data >> 8);
    WriteData((unsigned char) Data);
}


// Initialize LCD
void LCD_InitMain()
{
#ifdef STANDARD_18
    WriteCom(0x36);
    WriteData(0xC0);

    WriteCom(0x3a);
    WriteData(0x05);

    WriteCom(0xC0); //Set VRH1[4:0] & VC[2:0] for VCI1 & GVDD
    WriteData(0x08);
    WriteData(0x00);

    WriteCom(0xC1); //Set BT[2:0] for AVDD & VCL & VGH & VGL
    WriteData(0x03);

    WriteCom(0xC2);
    WriteData(0x05);

    WriteCom(0xC5); //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
    WriteData(0x43);
    WriteData(0x43);

    WriteCom(0xC7);
    WriteData(0xC5);

    WriteCom(0xB1);
    WriteData(0x08);
    WriteData(0x14);

    //WriteCom(0xB4);
    //WriteData(0x06);

    WriteCom(0xEC); //Set pumping clock frequence
    WriteData(0x0C);

    WriteCom(0xf2); //Enable Gamma bit
    WriteData(0x01);

    WriteCom(0xE0);
    WriteData(0x3E); //p63
    WriteData(0x1C); //p62
    WriteData(0x29); //p61
    WriteData(0x24); //p59
    WriteData(0x1D); //p57
    WriteData(0x09); //p50
    WriteData(0x50); //p43
    WriteData(0xC8); //p27/36
    WriteData(0x42); //p20
    WriteData(0x19); //p13
    WriteData(0x1C); //p6
    WriteData(0x0F); //p4
    WriteData(0x0E); //p2
    WriteData(0x05); //p1
    WriteData(0x00); //p0
    WriteCom(0xE1);
    WriteData(0x01); //p63
    WriteData(0x06); //p62
    WriteData(0x19); //p61
    WriteData(0x0B); //p59
    WriteData(0x12); //p57
    WriteData(0x10); //p50
    WriteData(0x27); //p43
    WriteData(0x58); //p27/36
    WriteData(0x3A); //p20
    WriteData(0x08); //p13
    WriteData(0x1A); //p6
    WriteData(0x1E); //p4
    WriteData(0x27); //p2
    WriteData(0x3A); //p1
    WriteData(0x3F); //p0
    
    WriteCom(0x11); //Exit Sleep
    Delay_ms(120);
    WriteCom(0x29); // Display On

#else /* DT010TFT */
    WriteCom(0x36);
    WriteData(0x08);

    WriteCom(0x3a); //Set Color Format
    WriteData(0x55);

    WriteCom(0x2A); //Set Column Address
    WriteData(0x00);
    WriteData(0x18);
    WriteData(0x00);
    WriteData(0x67);

    WriteCom(0x2B); //Set Page Address
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x9F);

    WriteCom(0xB1);//Set Frame Rate
    WriteData(0x0A);
    WriteData(0x05);

    WriteCom(0xC0); //Set VRH1[4:0] & VC[2:0] for VCI1 & GVDD
    WriteData(0x08);
    WriteData(0x00);

    WriteCom(0xC1); //Set BT[2:0] for AVDD & VCL & VGH & VGL
    WriteData(0x03);

    WriteCom(0xC2);
    WriteData(0x05);

    WriteCom(0xC5); //Set VMH[6:0] & VML[6:0] for VOMH & VCOML
    WriteData(0x43);
    WriteData(0x43);

    WriteCom(0xEC); //Set pumping clock frequence
    WriteData(0x0C);

    WriteCom(0xf2); //Enable Gamma bit
    WriteData(0x01);

    WriteCom(0xE0);
    WriteData(0x3B);//p1
    WriteData(0x2B);//p2
    WriteData(0x2A);//p3
    WriteData(0x2D);//p4
    WriteData(0x28);//p5
    WriteData(0x0C);//p6
    WriteData(0x57);//p7
    WriteData(0xF0);//p8
    WriteData(0x41);//p9
    WriteData(0x12);//p10
    WriteData(0x15);//p11
    WriteData(0x12);//p12
    WriteData(0x11);//p13
    WriteData(0x10);//p14
    WriteData(0x04);//p15

    WriteCom(0xE1);
    WriteData(0x04);//p1
    WriteData(0x14);//p2
    WriteData(0x15);//p3
    WriteData(0x12);//p4
    WriteData(0x17);//p5
    WriteData(0x13);//p6
    WriteData(0x28);//p7
    WriteData(0x60);//p8
    WriteData(0x3E);//p9
    WriteData(0x0D);//p10
    WriteData(0x2A);//p11
    WriteData(0x2D);//p12
    WriteData(0x2E);//p13
    WriteData(0x2F);//p14
    WriteData(0x3B);//p15

    WriteCom(0x11); //Exit Sleep
    Delay_ms(120);
    WriteCom(0x29); // Display On
#endif
}


void Delay_ms(unsigned int count)//1ms
{
    int i, j;
    for (i = 0; i < count; i++)
    {
        for (j = 0; j < 8900; j++);
    }
}

void ResetLCM()
{
    ResetHigh();
    Delay_ms(120);
    ResetLow();
    Delay_ms(80);
    ResetHigh();
    Delay_ms(80);
}

//****************************************************//
//********************LCD TEST************************//
//****************************************************//

void BlockWrite(unsigned int Xstart, unsigned int Xend, unsigned int Ystart, unsigned int Yend)
{
    WriteCom(0x2a);
    WriteData(Xstart >> 8);
    WriteData(Xstart + 24);
    WriteData(Xend >> 8);
    WriteData(Xend + 24);

    WriteCom(0x2b);
    WriteData(Ystart >> 8);
    WriteData(Ystart);
    WriteData(Yend >> 8);
    WriteData(Yend);

    WriteCom(0x2c);
}

void DispColor(unsigned int color)
{
    unsigned int i, j;

    BlockWrite(0, Lcm_H - 1, 0, Lcm_W - 1);

    CsLow();
    FourWireDataHigh();

    for (i = 0; i < Lcm_W; i++)
    {
        for (j = 0; j < Lcm_H; j++)
        {
            SpiWrite8Bits(color >> 8);
            SpiWrite8Bits(color);
        }
    }

    CsHigh();
}

void DispBand(void)
{
    unsigned int i, j, k;
    //unsigned int color[8]={0x001f,0x07e0,0xf800,0x07ff,0xf81f,0xffe0,0x0000,0xffff};
    unsigned int color[8] = {0xf800, 0xf800, 0x07e0, 0x07e0, 0x001f, 0x001f, 0xffff, 0xffff}; //0x94B2
    //unsigned int gray16[]={0x0000,0x1082,0x2104,0x3186,0x42,0x08,0x528a,0x630c,0x738e,0x7bcf,0x9492,0xa514,0xb596,0xc618,0xd69a,0xe71c,0xffff};

    BlockWrite(0, Lcm_H - 1, 0, Lcm_W - 1);

    CsLow();
    FourWireDataHigh();

    for (i = 0; i < 8; i++)
    {
        for (j = 0; j < Lcm_W / 8; j++)
        {
            for (k = 0; k < Lcm_H; k++)
            {
                SpiWrite8Bits(color[i] >> 8);
                SpiWrite8Bits(color[i]);
            }
        }
    }

    for (j = 0; j < (Lcm_W % 8); j++)
    {
        for (k = 0; k < Lcm_H; k++)
        {
            SpiWrite8Bits(color[7] >> 8);
            SpiWrite8Bits(color[7]);
        }
    }

    CsHigh();
}
